<?php

declare(strict_types=1);

namespace Drupal\group_sitemap\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\group\Entity\GroupInterface;
use Drupal\group_sitemap\SiteMap;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Group Sitemap Controller.
 */
final class GroupSitemapController extends ControllerBase {

  /**
   * Constructs a new GroupSitemapController object.
   */
  public function __construct(
    private readonly ModuleExtensionList $moduleExtensionList,
    private readonly SiteMap $siteMap,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('extension.list.module'),
      $container->get('group_sitemap.sitemap')
    );
  }

  /**
   * Return the sitemap.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   An XML response.
   */
  public function sitemap(GroupInterface $group): Response {
    $output = $this->siteMap->generate($group);
    if ($output === '') {
      throw new NotFoundHttpException();
    }

    return new Response($output, Response::HTTP_OK, [
      'Content-type' => 'application/xml; charset=utf-8',
      'X-Robots-Tag' => 'noindex, follow',
    ]);
  }

  /**
   * Provides sitemap XSL.
   */
  public function renderSitemapXsl(): Response {

    // Read the XSL content from the file.
    $module_path = $this->moduleExtensionList->getPath('group_sitemap');
    $xsl_content = \file_get_contents($module_path . '/xsl/xmlsitemap.xsl');
    if ($xsl_content === FALSE) {
      throw new NotFoundHttpException();
    }

    // Make sure the strings in the XSL content are translated properly.
    $replacements = [
      'Sitemap file' => $this->t('Sitemap file'),
      'Generated by the <a href="https://www.drupal.org/project/group_sitemap">Drupal Group Sitemap module</a>.' => $this->t('Generated by the <a href="@link-sitemap">Drupal Group Sitemap module</a>.', ['@link-sitemap' => 'https://www.drupal.org/project/group_sitemap']),
      'Number of sitemaps in this index' => $this->t('Number of sitemaps in this index'),
      'Click on the table headers to change sorting.' => $this->t('Click on the table headers to change sorting.'),
      'Sitemap URL' => $this->t('Sitemap URL'),
      'Last modification date' => $this->t('Last modification date'),
      'Number of URLs in this sitemap' => $this->t('Number of URLs in this sitemap'),
      'URL location' => $this->t('URL location'),
      'Change frequency' => $this->t('Change frequency'),
      'Priority' => $this->t('Priority'),
      '[jquery]' => \base_path() . 'core/assets/vendor/jquery/jquery.js',
      '[jquery-tablesort]' => \base_path() . $module_path . '/xsl/jquery.tablesorter.min.js',
      '[xsl-js]' => \base_path() . $module_path . '/xsl/xmlsitemap.xsl.js',
      '[xsl-css]' => \base_path() . $module_path . '/xsl/xmlsitemap.xsl.css',
    ];
    $xsl_content = \strtr($xsl_content, $replacements);

    // Output the XSL content.
    return new Response($xsl_content, 200, [
      'Content-Type' => 'application/xml; charset=utf-8',
      'X-Robots-Tag' => 'noindex, nofollow',
    ]);
  }

}
