<?php

declare(strict_types=1);

namespace Drupal\group_sitemap;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AnonymousUserSession;
use Drupal\Core\Site\Settings;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Entity\GroupRelationshipInterface;
use Drupal\group_sitemap\Entity\GroupSitemap;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\Attribute\AutowireCallable;

/**
 * Sitemap utility service class.
 */
final class SiteMap {

  public const QUEUE_NAME = 'group_sitemap_processor';
  public const RECREATE_STATE_KEY = 'group_sitemap_enqueue_all';

  private const MAX_LINKS = 50000;
  private const CACHE_PREFIX = 'group_sitemap';

  /**
   * Array of affected bundles keyed by entity type ID.
   */
  private ?array $invalidateData = NULL;

  /**
   * Constructor.
   */
  public function __construct(
    #[Autowire(service: 'cache.data')]
    protected readonly CacheBackendInterface $cache,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly SiteMapItemStorage $itemStorage,
    #[AutowireCallable(service: 'date.formatter', method: 'format', lazy: TRUE)]
    protected \Closure $formatDate,
    #[AutowireCallable(service: 'group_relation_type.manager', method: 'getPluginIdsByEntityTypeId', lazy: TRUE)]
    protected \Closure $getPluginIdsByEntityTypeId,
    #[Autowire(service: 'logger.channel.group_sitemap', lazy: TRUE)]
    protected readonly LoggerChannelInterface $logger,
    #[Autowire(service: 'state', lazy: TRUE)]
    protected readonly StateInterface $state,
    #[AutowireCallable(service: 'queue', method: 'get', lazy: TRUE)]
    protected \Closure $getQueue,
  ) {}

  /**
   * Get entity types and bundles that are included in group site maps.
   */
  private function getInvalidateData(): array {
    if ($this->invalidateData !== NULL) {
      return $this->invalidateData;
    }

    $this->invalidateData = [
      'bundles' => [],
      'group_types' => [],
      'relationship_types' => [],
    ];
    foreach ($this->entityTypeManager->getStorage('group_sitemap')->loadByProperties([
      'status' => TRUE,
    ]) as $group_type => $config_entity) {
      \assert($config_entity instanceof GroupSitemap);

      $this->invalidateData['group_types'][$group_type] = $group_type;

      foreach ($config_entity->getRelationshipData() as $data_item) {
        $this->invalidateData['relationship_types'][$data_item['relationship_type_id']] = $data_item['relationship_type_id'];

        if (!\array_key_exists($data_item['entity_type_id'], $this->invalidateData['bundles'])) {
          $this->invalidateData['bundles'][$data_item['entity_type_id']] = [];
        }
        $this->invalidateData['bundles'][$data_item['entity_type_id']][$data_item['entity_bundle']] = $data_item['entity_bundle'];
      }
    }

    return $this->invalidateData;
  }

  /**
   * Shortcut if we have ID only.
   */
  public function setGroupItemFromId(string $group_id): void {
    $group = $this->entityTypeManager->getStorage('group')->load($group_id);
    if ($group === NULL) {
      return;
    }
    $this->setGroupItem($group);
  }

  /**
   * Set a single group root sitemap link item.
   */
  public function setGroupItem(GroupInterface $group): void {
    $config_entity = $this->entityTypeManager->getStorage('group_sitemap')->load($group->bundle());
    if (
      $config_entity === NULL ||
      !$config_entity instanceof GroupSitemap ||
      !$config_entity->status()
    ) {
      $this->itemStorage->deleteItem((string) $group->id());
      return;
    }

    $this->itemStorage->mergeItem([
      'group_id' => $group->id(),
      'relationship_id' => 0,
      'loc' => $group->toUrl('canonical', [
        'absolute' => TRUE,
      ])->toString(),
      'lastmod' => ($this->formatDate)(
        $group->getChangedTime(),
        'custom',
        'Y-m-d'
      ),
      'changefreq' => $config_entity->homeChangeFreq(),
      'priority' => '1',
    ]);
  }

  /**
   * Set a single group relationship sitemap link item.
   */
  public function setGroupRelationshipItem(GroupRelationshipInterface $relationship): void {
    // Skip orphaned relationships.
    $group = $relationship->getGroup();
    if (!$group instanceof GroupInterface) {
      $this->itemStorage->deleteItem((string) $relationship->getGroupId(), (string) $relationship->id());
      return;
    }

    $config_entity = $this->entityTypeManager->getStorage('group_sitemap')->load($group->bundle());
    if (
      $config_entity === NULL ||
      !$config_entity instanceof GroupSitemap ||
      !$config_entity->status()
    ) {
      $this->itemStorage->deleteItem((string) $relationship->getGroupId(), (string) $relationship->id());
      return;
    }

    // Free some memory.
    unset($group);

    $relationship_bundle = $relationship->bundle();
    $type_config = NULL;
    foreach ($config_entity->getRelationshipData() as $data_item) {
      if ($data_item['relationship_type_id'] === $relationship_bundle) {
        $type_config = $data_item;
        break;
      }
    }
    if ($type_config === NULL) {
      $this->itemStorage->deleteItem((string) $relationship->getGroupId(), (string) $relationship->id());
      return;
    }

    $anonymous = new AnonymousUserSession();
    if (!$relationship->access('view', $anonymous)) {
      $this->itemStorage->deleteItem((string) $relationship->getGroupId(), (string) $relationship->id());
      return;
    }

    $entity = $relationship->getEntity();
    if ($entity === NULL || !$entity->access('view', $anonymous)) {
      $this->itemStorage->deleteItem((string) $relationship->getGroupId(), (string) $relationship->id());
      return;
    }

    if ($entity instanceof EntityChangedInterface) {
      $lastmod_ts = $entity->getChangedTime();
    }
    else {
      $lastmod_ts = $relationship->getChangedTime();
    }

    $this->itemStorage->mergeItem([
      'group_id' => $relationship->getGroupId(),
      'relationship_id' => $relationship->id(),
      'loc' => $relationship->toUrl('canonical', [
        'absolute' => TRUE,
      ])->toString(),
      'lastmod' => ($this->formatDate)(
        $lastmod_ts,
        'custom',
        'Y-m-d'
      ),
      'changefreq' => $type_config['change_frequency'],
      'priority' => (string) $type_config['priority'],
    ]);
  }

  /**
   * Generate a sitemap for a group.
   */
  public function generate(GroupInterface $group): string {
    $cid = self::CACHE_PREFIX . ':' . $group->id();
    $cached = $this->cache->get($cid);
    if ($cached !== FALSE) {
      return $cached->data;
    }

    $links = $this->itemStorage->getLinks((string) $group->id());
    if (\count($links) === 0) {
      return '';
    }

    $sitemap = $this->generateXml($links, (string) $group->id());
    $this->cache->set($cid, $sitemap, CacheBackendInterface::CACHE_PERMANENT);
    return $sitemap;
  }

  /**
   * Rebuild links for an entity.
   */
  public function rebuildItemsForEntity(EntityInterface $entity): void {
    if (!$entity instanceof ContentEntityInterface) {
      return;
    }
    if ($entity instanceof GroupInterface) {
      $this->setGroupItem($entity);
      $this->cache->invalidate(self::CACHE_PREFIX . ':' . $entity->id());
      return;
    }
    if ($entity instanceof GroupRelationshipInterface) {
      $this->setGroupRelationshipItem($entity);
      $this->cache->invalidate(self::CACHE_PREFIX . ':' . $entity->getGroupId());
      return;
    }

    $invalidate_data = $this->getInvalidateData();
    if (
      \count($invalidate_data['group_types']) === 0 ||
      \count($invalidate_data['relationship_types']) === 0
    ) {
      return;
    }

    // Return early if this entity type or bundle is not included in group
    // sitemaps.
    $entity_type_id = $entity->getEntityTypeId();
    if (
      !\array_key_exists($entity_type_id, $invalidate_data['bundles']) ||
      !\array_key_exists($entity->bundle(), $invalidate_data['bundles'][$entity_type_id])
    ) {
      return;
    }

    $plugin_ids = ($this->getPluginIdsByEntityTypeId)($entity->getEntityTypeId());
    if (\count($plugin_ids) === 0) {
      return;
    }

    $group_relationship_storage = $this->entityTypeManager->getStorage('group_relationship');
    $query = $group_relationship_storage->getQuery();
    $results = $query
      ->accessCheck(FALSE)
      ->condition('entity_id', $entity->id())
      ->condition('plugin_id', $plugin_ids, 'IN')
      ->condition('type', $invalidate_data['relationship_types'], 'IN')
      ->condition('group_type', $invalidate_data['group_types'], 'IN')
      ->execute();
    if (\count($results) === 0) {
      return;
    }

    $cids = [];
    foreach ($group_relationship_storage->loadMultiple($results) as $group_relationship) {
      $this->setGroupRelationshipItem($group_relationship);
      $gid = $group_relationship->getGroupId();
      $cids[$gid] = self::CACHE_PREFIX . ':' . $gid;
    }
    $this->cache->invalidateMultiple($cids);
  }

  /**
   * Generate a sitemap xml given an array of links.
   *
   * @param array<array[string]> $links
   *   Array of arrays containing xml sitemap url parameters:
   *   - loc
   *   - lastmod
   *   - ..
   * @param string $group_id
   *   Group ID for error logging purposes.
   */
  private function generateXml(array $links, string $group_id): string {
    $xml = new \XMLWriter();
    $xml->openMemory();
    // @todo Set this to false when development is done.
    $xml->setIndent(TRUE);
    $this->startDocument($xml);

    // @todo Split to chunks with an index xml if ever needed.
    if (\count($links) > self::MAX_LINKS) {
      $this->logger->warning(\sprintf('Maximum number of links reached for group %d.', $group_id));
      $links = \array_slice($links, 0, self::MAX_LINKS);
    }
    foreach ($links as $link) {
      $xml->startElement('url');
      foreach ([
        'loc',
        'lastmod',
        'changefreq',
        'priority',
      ] as $parameter) {
        $xml->startElement($parameter);
        if (\array_key_exists($parameter, $link)) {
          $xml->text($link[$parameter]);
        }
        $xml->endElement();
      }
      $xml->endElement();
    }

    $xml->endDocument();
    return (string) $xml->flush();
  }

  /**
   * Start XML document.
   */
  private function startDocument(\XMLWriter $xml): void {
    $xml->startDocument('1.0', 'UTF-8');
    $xls_url = Url::fromRoute('group_sitemap.sitemap_xsl', [], [
      'absolute' => TRUE,
    ])->toString();
    $xml->writePi('xml-stylesheet', 'type="text/xsl" href="' . $xls_url . '"');
    $xml->startElement('urlset');

    $attributes = [];
    $attributes['xmlns'] = 'http://www.sitemaps.org/schemas/sitemap/0.9';
    $attributes['xmlns:xhtml'] = 'http://www.w3.org/1999/xhtml';

    foreach ($attributes as $key => $value) {
      $xml->writeAttribute($key, $value);
    }
  }

  /**
   * Enqueue a batch of sitemap items for regeneration.
   */
  public function enqueueBatch(array &$sandbox): void {
    if (!\array_key_exists('list', $sandbox)) {
      $last_id = $this->state->get(self::RECREATE_STATE_KEY);
      if ($last_id === NULL) {
        $last_id = ['group_relationship', 0];
      }

      if ($last_id[0] === 'group_relationship' && $last_id[1] === 0) {
        $this->itemStorage->deleteAll();
      }

      $sandbox['list'] = $this->getBatch($last_id, Settings::get('group_sitemap_enqueue_batch_size', 1000));

      if (\count($sandbox['list']) === 0) {
        $this->state->delete(self::RECREATE_STATE_KEY);
      }
      else {
        $this->state->set(self::RECREATE_STATE_KEY, $last_id);
      }

      $sandbox['total'] = \count($sandbox['list']);
      $sandbox['processed'] = 0;
    }

    $item = \array_pop($sandbox['list']);
    if ($item === NULL) {
      return;
    }
    ($this->getQueue)(self::QUEUE_NAME)->createItem($item);
    $sandbox['processed']++;
    $sandbox['finished'] = $sandbox['processed'] / $sandbox['total'];
  }

  /**
   * Batch API logic.
   */
  public static function getBatchArray(): array {
    $batch_builder = (new BatchBuilder())
      ->setTitle(new TranslatableMarkup('Indexing group sitemap content'))
      ->setInitMessage(new TranslatableMarkup('Initializing..'))
      ->setProgressMessage(new TranslatableMarkup('Indexing..'))
      ->setErrorMessage(new TranslatableMarkup('An error occurred while indexing group sitemap content.'));

    $batch_builder->addOperation([SiteMap::class, 'indexOperation'], []);
    return $batch_builder->toArray();
  }

  /**
   * Index batch operation.
   */
  public static function indexOperation(array &$context): void {
    $service = \Drupal::service('group_sitemap.sitemap');

    if (!\array_key_exists('total', $context['sandbox'])) {
      \Drupal::service('group_sitemap.item_storage')->deleteAll();
      $context['sandbox']['total'] = $service->getTotalCount();
      $context['sandbox']['current'] = ['group_relationship', '0'];
      $context['sandbox']['processed'] = 0;
    }

    // Process in batches.
    $list = $service->getBatch($context['sandbox']['current'], Settings::get('entity_update_batch_size', 50));
    if (\count($list) === 0) {
      $context['finished'] = 1;
      return;
    }

    foreach ($list as $item) {
      $service->processItem($item);
      $context['sandbox']['processed']++;
    }

    // It's possible new items are added during batch process so we cannot
    // rely on $sandbox['total'] but rather finish when there's nothing
    // else to process.
    $context['finished'] = 0;
    $context['results']['count'] = $context['sandbox']['processed'];
    $context['message'] = new TranslatableMarkup('Processed @count of @total entities.', [
      '@count' => $context['sandbox']['processed'],
      '@total' => $context['sandbox']['total'],
    ]);
  }

  /**
   * Get total number of items to process in batch operations.
   */
  public function getTotalCount(): int {
    $total_items = 0;
    $invalidate_data = $this->getInvalidateData();

    $query = $this->entityTypeManager->getStorage('group')->getQuery();
    $total_items += $query
      ->accessCheck(FALSE)
      ->condition('type', $invalidate_data['group_types'], 'IN')
      ->count()
      ->execute();

    $query = $this->entityTypeManager->getStorage('group_relationship')->getQuery();
    $total_items += $query
      ->accessCheck(FALSE)
      ->condition('type', $invalidate_data['relationship_types'], 'IN')
      ->count()
      ->execute();

    return $total_items;
  }

  /**
   * Get batch of entity IDs to enqueue.
   */
  public function getBatch(array &$selection_data, int $max_items): array {
    $invalidate_data = $this->getInvalidateData();
    $list = [];

    // Enqueue group relationship entities first so groups itself will
    // always be processed last and we can invalidate site maps cache
    // only once when all else has been indexed.
    if ($selection_data[0] === 'group_relationship') {
      $query = $this->entityTypeManager->getStorage('group_relationship')->getQuery();
      $relationship_ids = $query
        ->accessCheck(FALSE)
        ->condition('type', $invalidate_data['relationship_types'], 'IN')
        ->condition('id', $selection_data[1], '>')
        ->range(0, $max_items)
        ->execute();
      if (\count($relationship_ids) === 0) {
        $selection_data[0] = 'group';
        $selection_data[1] = 0;
      }
      else {
        foreach ($relationship_ids as $relationship_id) {
          $list[] = ['group_relationship', $relationship_id];
        }
        // Phpstan: "Variable $relationship_id might not be defined."
        // - that's not true.
        // @phpstan-ignore-next-line
        $selection_data[1] = $relationship_id;
      }
    }

    if ($selection_data[0] === 'group') {
      $query = $this->entityTypeManager->getStorage('group')->getQuery();
      $group_ids = $query
        ->accessCheck(FALSE)
        ->condition('type', $invalidate_data['group_types'], 'IN')
        ->condition('id', $selection_data[1], '>')
        ->range(0, $max_items)
        ->execute();

      if (\count($group_ids) === 0) {
        return $list;
      }
      else {
        foreach ($group_ids as $group_id) {
          $list[] = ['group', $group_id];
        }
        // Phpstan: "Variable $group_id might not be defined."
        // - that's not true.
        // @phpstan-ignore-next-line
        $selection_data[1] = $group_id;
      }
    }

    return $list;
  }

  /**
   * Process a queue / batch item.
   */
  public function processItem(array $item): void {
    $entity = $this->entityTypeManager->getStorage($item[0])->load($item[1]);
    if ($entity === NULL) {
      return;
    }
    if ($item[0] === 'group') {
      \assert($entity instanceof GroupInterface);
      $this->setGroupItem($entity);
      // Since we're operating on all items, it's enough to invalidate
      // for groups that run last in the queues.
      $this->cache->invalidate(self::CACHE_PREFIX . ':' . $entity->id());
    }
    elseif ($item[0] === 'group_relationship') {
      \assert($entity instanceof GroupRelationshipInterface);
      $this->setGroupRelationshipItem($entity);
    }
  }

}
