<?php

declare(strict_types=1);

namespace Drupal\group_sitemap;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\StatementInterface;

/**
 * Site map items storage handler class.
 */
final class SiteMapItemStorage {

  public const TABLE = 'group_sitemap_items';

  /**
   * The obvious.
   */
  public function __construct(
    private readonly Connection $db,
  ) {}

  /**
   * Get all sitemap links for a group.
   */
  public function getLinks(string $group_id): array {
    $statement = $this->db->select(self::TABLE)
      ->fields(self::TABLE)
      ->condition('group_id', $group_id)
      ->orderBy('priority', 'DESC')
      ->execute();
    \assert($statement instanceof StatementInterface);
    return $statement->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Create / update item.
   */
  public function mergeItem(array $item): void {
    $this->db->merge(self::TABLE)
      ->key('group_id', $item['group_id'])
      ->key('relationship_id', $item['relationship_id'])
      ->fields($item)
      ->execute();
  }

  /**
   * Delete item or all items with a given group ID.
   */
  public function deleteItem(string $group_id, ?string $relationship_id = '0'): void {
    $statement = $this->db->delete(self::TABLE)
      ->condition('group_id', $group_id);
    if ($relationship_id !== NULL) {
      $statement->condition('relationship_id', $relationship_id);
    }
    $statement->execute();
  }

  /**
   * Clear the entire sitemap items data.
   */
  public function deleteAll(): void {
    $this->db->delete(self::TABLE)->execute();
  }

}
