<?php

namespace Drupal\Tests\groupmedia\Kernel;

use Drupal\Tests\group\Kernel\GroupKernelTestBase;

/**
 * Tests tracking behaviour.
 *
 * @group groupmedia
 */
class GroupMediaTestTracking extends GroupKernelTestBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The group we will use to test methods on.
   *
   * @var \Drupal\group\Entity\Group
   */
  protected $group;

  /**
   * The media group relationship type.
   *
   * @var \Drupal\group\Entity\GroupRelationshipType
   */
  protected $groupRelationshipType;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'file',
    'image',
    'text',
    'system',
    'node',
    'gnode',
    'media',
    'groupmedia',
    'groupmedia_test_config',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('file');
    $this->installEntitySchema('media');
    $this->installEntitySchema('node');

    $this->installSchema('file', 'file_usage');
    $this->installSchema('node', ['node_access']);

    $this->installConfig([
      'field',
      'system',
      'image',
      'file',
      'media',
      'groupmedia_test_config',
    ]);

    $this->group = $this->createGroup();

    $group_type = $this->group->getGroupType();

    // Enable groupmedia remote video plugin and node plugin.
    $group_content_type_storage = $this->entityTypeManager->getStorage('group_content_type');
    $group_content_type_storage->save($group_content_type_storage->createFromPlugin($group_type, 'group_media:remote_video'));
    $group_content_type_storage->save($group_content_type_storage->createFromPlugin($group_type, 'group_node:group_media_content'));

    $this->pluginManager->clearCachedPluginMaps();

    $this->groupRelationshipType = $group_content_type_storage->load($group_content_type_storage->getRelationshipTypeId($group_type->id(), 'group_media:remote_video'));
  }

  /**
   * Test that media is added to group, when tracking is enabled.
   */
  public function testContentAddingWithEnabledTracking() {
    $media = $this->createMedia();
    $node = $this->createNode(['field_media' => $media->id()]);
    $this->group->addRelationship($node, 'group_node:group_media_content');

    $this->assertEquals(0, count($this->group->getRelationships('group_media:remote_video')));

    $configuration = $this->groupRelationshipType->get('plugin_config');
    $configuration['tracking_enabled'] = 1;
    $this->groupRelationshipType->set('plugin_config', $configuration);
    $this->groupRelationshipType->save();

    $this->pluginManager->clearCachedPluginMaps();

    $node = $this->createNode(['field_media' => $media->id()]);
    $this->group->addRelationship($node, 'group_node:group_media_content');

    $this->assertEquals(1, count($this->group->getRelationships('group_media:remote_video')));
  }

  /**
   * Create a media item.
   *
   * @param array $values
   *   Additional properties.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   Media item.
   */
  protected function createMedia(array $values = []) {
    $storage = $this->entityTypeManager->getStorage('media');
    $media = $storage->create($values + [
      'bundle' => 'remote_video',
      'name' => $this->randomString(),
      'field_media_oembed_video' => 'https://www.youtube.com/watch?v=dQw4w9WgXcQ',
    ]);
    $media->enforceIsNew();
    $storage->save($media);
    return $media;
  }

  /**
   * Create a node item.
   *
   * @param array $values
   *   Additional properties.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   Media item.
   */
  protected function createNode(array $values = []) {
    $storage = $this->entityTypeManager->getStorage('node');
    $node = $storage->create($values + [
      'type' => 'group_media_content',
      'title' => $this->randomString(),
    ]);
    $node->enforceIsNew();
    $storage->save($node);
    return $node;
  }

}
