<?php

/**
 * @file
 * Contains the hooks created by this module.
 *
 * All hooks implemented in this module may reside in a MODULE.gsislogin.inc
 * file.
 */

/**
 * Alters the url to redirect upon GSIS failure.
 *
 * @param \Drupal\Core\Url $url
 *   The url to redirect to. Any value valid for url() is accepted.
 */
function hook_gsislogin_failure_redirect_url_alter(\Drupal\Core\Url &$url) {
  $url = \Drupal\Core\Url::fromRoute('user.login');
}

/**
 * Alters the url to redirect upon GSIS success login/registration.
 *
 * @param string $url
 *   The url to redirect to. Any value valid for url() is accepted.
 */
function hook_gsislogin_success_redirect_url_alter(\Drupal\Core\Url &$url) {
  $url = \Drupal\Core\Url::fromRoute('<front>');
}

/**
 * Allows modules to change the redirect uri's query parameters.
 *
 * Parameters passed in this hook will also be provided to the next hooks too.
 *
 * @param array $query
 *   The query arguments to be sent back to the website.
 */
function hook_gsislogin_redirect_uri_query_alter(array &$query) {
  $query['foo'] = 'bar';
}

/**
 * Allows modules to say if the user with the provided details exists.
 *
 * @param array $details
 *   An array with the values retrieved from GSIS.
 * @param array $params
 *   An array with the query parameters previously passed to the redirect uri.
 *
 * @return int|void
 *   Return the uid of the user if exists. Return nothing if you're not
 *   responsible for deciding.
 */
function hook_gsislogin_user_exists(array $details, array $params) {
  return \Drupal::database()->select('users_field_data', 'u')
    ->fields('u', ['uid'])
    ->condition('u.name', $details['userid'])
    ->execute()
    ->fetchField();
}

/**
 * Allows modules to say if the user with the provided data should be created.
 *
 * This hook is only invoked if the user with the $details doesn't already
 * exist.
 *
 * @param array $details
 *   The details retrieved by the GSIS system.
 * @param array $params
 *   An array with the query parameters previously passed to the redirect uri.
 *
 * @return bool|void
 *   Return FALSE if the user should not be created. Return nothing if you
 *   aren't responsible for deciding. If all modules return nothing, the user
 *   will be created.
 */
function hook_gsislogin_user_create(array $details, array $params) {
  // Dummy code checking the last name of the user in GSIS.
  if (mb_strlen($details['lastname']) > 10) {
    return TRUE;
  }
}

/**
 * Allows modules to change the namespace used for the external login.
 *
 * @param string $module
 *   The module namespace. Defaults to 'gsislogin'.
 * @param array $context
 *   The array with the context to modify. Keys used by this module are:
 *   - username:
 *     The username checked or created by the authentication mechanism.
 *   - details:
 *     The user details retrieved from the GSIS service.
 *   - request:
 *     The request object in case query arguments need to be checked also.
 */
function hook_gsislogin_user_module_alter(&$module, &$context) {
  if ($context['username'] === 'foo') {
    $context['username'] = 'bar';
  }
}

/**
 * Allows modules to act upon a user created/logged-in.
 *
 * @param object $account
 *   The user entity to be processed.
 * @param array $details
 *   The user details retrieved by the GSIS system.
 * @param array $params
 *   An array with the query parameters previously passed to the redirect uri.
 */
function hook_gsislogin_create_user_details($account, array $details, array $params) {
  // Meaningless code. If user is authenticated, also gets the role 'administrator'.
  if (Drupal::currentUser()->isAuthenticated()) {
    $user = \Drupal\user\Entity\User::load(Drupal::currentUser()->id());
    $user->addRole('administrator');
    $user->save();
  }
}
