<?php

namespace Drupal\gsislogin\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use GuzzleHttp\Exception\ClientException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Drupal\Component\Serialization\Json;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Language\LanguageManager;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\user\Entity\User;

/**
 * Controller for handling GSIS OAuth2 login integration.
 */
class GsisLoginController extends ControllerBase {

  protected $tag = "_gsis_";
  protected $appId = "";
  protected $secret = "";
  protected $redirectUri = "";
  protected $tokenUrl = "";
  protected $userinfoUrl = "";
  protected $authorizeUrl = "";
  protected $config = NULL;
  protected $language = NULL;
  protected $httpClient = NULL;

  /**
   * Constructor for GsisLoginController.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory to retrieve configuration objects.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack object.
   * @param \GuzzleHttp\Client $http_client
   *   The http client.
   * @param \Drupal\Core\Language\LanguageManager $language_manager
   *   The language manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, RequestStack $request_stack, Client $http_client, LanguageManager $language_manager) {
    $request = $request_stack->getCurrentRequest();
    $this->language = $language_manager->getCurrentLanguage()->getId();
    $this->httpClient = $http_client;
    $this->config = $config_factory->get('config.gsislogin');
    $this->tag = "_gsis_" . uniqid(mt_rand(), TRUE);
    $this->appId = $this->config->get('GSISID');
    $this->secret = $this->config->get('GSISSECRET');
    $this->redirectUri = $request->getScheme() . "://" . $request->getHost() . "/gsis";
    $this->tokenUrl = 'https://oauth2.gsis.gr/oauth2server/oauth/token';
    $this->userinfoUrl = 'https://oauth2.gsis.gr/oauth2server/userinfo';
    $this->authorizeUrl = 'https://oauth2.gsis.gr/oauth2server/oauth/authorize';
    if ($this->config->get('GSISTEST')) {
      $this->tokenUrl = 'https://test.gsis.gr/oauth2server/oauth/token';
      $this->userinfoUrl = 'https://test.gsis.gr/oauth2server/userinfo';
      $this->authorizeUrl = 'https://test.gsis.gr/oauth2server/oauth/authorize';
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('request_stack'),
      $container->get('http_client'),
      $container->get('language_manager')
    );
  }

  /**
   * Implements the GSIS OAuth2 login process.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request object.
   *
   * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
   *   Returns a render array with a message in case of error, or a redirect
   *   response to the GSIS authorization URL or post-login destination.
   */
  public function start(Request $request) {
    // Set session if not already set.
    $session = $request->getSession();
    if (!$session->has('gsis_state')) {
      $session->set('gsis_state', $this->tag);
    }

    // Set the redirect url and allow other modules to add parameters to it.
    $query = [];
    // Allow modules to change the redirect uri's query params.
    $this->moduleHandler()->alter('gsislogin_redirect_uri_query', $query);
    $locked_parameters = ['code', 'error', 'error_description', 'state'];
    foreach ($locked_parameters as $parameter) {
      if (isset($query[$parameter])) {
        unset($query[$parameter]);
        throw new \InvalidArgumentException(sprintf(
          'A module attempted to add the locked parameter "%s" to the query string. Locked parameters are: %s.',
          $parameter,
          implode(',', $locked_parameters)
        ));
      }
    }

    // In case of GSIS error.
    if (NULL !== $request->get('error')) {
      if (NULL !== $request->get('error_description')) {
        if ($request->get('error_description') === "User denied access") {
          return ['#markup' => $this->t("You chose not to proceed to login from the systems of the Greek General Secretariat for Information Systems.")];
        }
        return ['#markup' => $this->t("ERR: 001 There was a problem connecting to the systems of the General Secretariat for Information Systems.")];
      }
      else {
        $url = Url::fromRoute('user.login');
        // Allow other modules to change the redirect url upon failure.
        $this->moduleHandler()->alter('gsislogin_failure_redirect_url', $url);
        return new RedirectResponse($url->toString());
      }
    }

    /*
     * Step #2.
     * Upon success get token with code.
     */
    if (NULL !== $request->get('code')) {

      $code = $request->get('code');
      $state = $request->get('state');

      if ($state != $session->get('gsis_state')) {
        return ['#markup' => $this->t("ERR: 002 Problem in response to the systems of the General Secretariat for Information Systems.")];
      }

      try {
        $gettokenurl = $this->tokenUrl;
        $data = [
          'code' => $code,
          'redirect_uri' => $this->redirectUri,
          'client_id' => $this->appId,
          'client_secret' => $this->secret,
          'scope' => '',
          'grant_type' => 'authorization_code',
        ];

        $body = http_build_query($data);

        $tokenjson = $this->httpClient->post(
          $gettokenurl, [
            'body' => $body,
            'headers' => [
              'Accept' => 'application/json',
              'Content-Type' => 'application/x-www-form-urlencoded',
            ],
          ]
        );
        $token = Json::decode($tokenjson->getBody()->getContents());
      }
      catch (RequestException $e) {
        $this->getLogger('gsislogin')->warning('GSIS request failed with exception: @message', [
          '@message' => $e->getMessage(),
        ]);
        return ['#markup' => $this->t("ERR: 003 Problem in connection with the General Secretariat of Information Systems.")];
      }

      if (isset($token['error'])) {
        $this->getLogger('gsislogin')->warning('Error decoding JSON: @json', [
          '@json' => (string) $tokenjson->getBody(),
        ]);
        return [
          '#markup' => t('ERR: 004 Data entry problem by the General Secretariat of Information Systems.'),
        ];
      }

      /*
       * Step #3.
       *
       * Calling httpClient didn't work; returned 400 every time.
       * Implemented fallback to simplexml_load_string().
       */
      try {
        $getuserurl = $this->userinfoUrl . "?format=xml&access_token=" . $token['access_token'];
        $userxml = $this->httpClient->get(
          $getuserurl, [
            'headers' => [
              'Accept' => 'application/xml',
              'Content-Type' => 'application/x-www-form-urlencoded;',
            ],
          ]
        )->getBody()->getContents();

        // In case of error, GSIS sends JSON !!!
        $checkerror = Json::decode($userxml);
        if ($checkerror !== NULL) {
          return [
            '#markup' => $this->t("ERR: 005 Data collection problem from the General Secretariat of Information Systems."),
          ];
        }

        $xml = simplexml_load_string($userxml);

        $userid = $xml->userinfo['userid'][0]->__toString();
        $taxid = $xml->userinfo['taxid'][0]->__toString();
        $lastname = $xml->userinfo['lastname'][0]->__toString();
        $firstname = $xml->userinfo['firstname'][0]->__toString();
        $fathername = $xml->userinfo['fathername'][0]->__toString();
        $mothername = $xml->userinfo['mothername'][0]->__toString();
        $birthyear = $xml->userinfo['birthyear'][0]->__toString();

        $userDetails = [
          "userid" => $userid,
          "taxid" => $taxid,
          "lastname" => $lastname,
          "firstname" => $firstname,
          "fathername" => $fathername,
          "mothername" => $mothername,
          "birthyear" => $birthyear,
        ];
        $response = $this->createUser($userDetails, $request);
        if ($response instanceof RedirectResponse) {
          return $response;
        }

        $url = Url::fromRoute('<front>');
        // Allow other modules to change the redirect url upon success.
        $this->moduleHandler()->alter('gsislogin_success_redirect_url', $url);
        return new RedirectResponse($url->toString());
      }
      catch (ClientException $e) {
        $this->getLogger('gsislogin')->error($e->getMessage());
        return ['#markup' => $this->t("ERR: 007 Data entry problem by the General Secretariat of Information Systems.")];
      }
      catch (RequestException $e) {
        $this->getLogger('gsislogin')->error($e->getMessage());
        return ['#markup' => $this->t("ERR: 006 Problem of entering data from the General Secretariat of Information Systems.")];
      }
    }

    /*
     * Step #1.
     */
    $query = [
      'client_id' => $this->appId,
      'redirect_uri' => $this->redirectUri,
      'response_type' => 'code',
      'scope' => 'read',
      'state' => $session->get('gsis_state'),
    ];
    return new TrustedRedirectResponse($this->authorizeUrl . '?' . http_build_query($query));
  }

  /**
   * Creates or logs in a Drupal user based on GSIS-provided details.
   *
   * @param array $userDetails
   *   The user details retrieved by GSIS.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse|null
   *   Returns a redirect response if the user cannot be created, otherwise
   *   NULL after successful creation/login.
   */
  protected function createUser(array $userDetails, Request $request) {
    // Allow modules to decide if the user exists.
    $uids = $this->moduleHandler()
      ->invokeAll('gsislogin_user_exists', [$userDetails, $request]);
    foreach ($uids as $uid) {
      if ($uid) {
        break;
      }
    }

    if (empty(array_filter($uids))) {
      // Allow modules to decide if the user should be created. If a module
      // returns FALSE, it should also call \Drupal::messenger() to ensure the
      // user understands why he couldn't get logged in.
      $created = $this->moduleHandler()
        ->invokeAll('gsislogin_user_create', [$userDetails, $request]);
      foreach ($created as $create) {
        if ($create === FALSE) {
          break;
        }
      }
      if (!$created[0]) {
        // If the user doesn't exist and a module thinks we should not create him,
        // we should fall back to the failure redirect url.
        $url = Url::fromRoute('user.login');
        // Allow other modules to change the redirect url upon failure.
        $this->moduleHandler()->alter('gsislogin_failure_redirect_url', $url);
        return new RedirectResponse($url->toString());
      }
    }
    else {
      $uid = reset($uids);
      $user_account = User::load($uid);
      $username = $user_account->getAccountName();
    }

    // If the user doesn't exist and needs to be created, we have to set its
    // username.
    if (!isset($username)) {
      $username = 'gsis_' . $userDetails['userid'];
    }

    // Allow other modules to change the login module namespace.
    // @todo The $module variable is not currently required. It may be needed in
    //   case we add externalauth contrib module as a dependency.
    $module = 'gsislogin';
    $context = [
      'username' => $username,
      'details' => $userDetails,
      'request' => $request,
    ];
    $this->moduleHandler()->alter('gsislogin_user_module', $module, $context);

    $user = user_load_by_name($username);
    if (!$user) {
      // Create the user.
      $user = $this->entityTypeManager()->getStorage('user')->create();
      $user->enforceIsNew();
      $user->setUsername($username);
      $user->activate();
    }

    // Update custom fields.
    $user->set('field_gsis_userid', $userDetails['userid']);
    $user->set('field_gsis_taxid', $userDetails['taxid']);
    $user->set('field_gsis_lastname', $userDetails['lastname']);
    $user->set('field_gsis_firstname', $userDetails['firstname']);
    $user->set('field_gsis_mothername', $userDetails['mothername']);
    $user->set('field_gsis_fathername', $userDetails['fathername']);
    $user->set('field_gsis_birthyear', $userDetails['birthyear']);
    $user->save();

    // Login the user.
    user_login_finalize($user);

    // Allow other modules to know that a user was created or logged-in by this
    // module specifically.
    $this->moduleHandler()->invokeAll('gsislogin_create_user_details', [
      $user,
      $userDetails,
      $request,
    ]);
    return NULL;
  }

}
