<?php

namespace Drupal\gutenberg_block_report\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\node\Entity\Node;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Render\Markup;
use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Controller for Gutenberg Block Report.
 */
class BlockReportController extends ControllerBase {

  protected $database;

  public function __construct(Connection $database) {
    $this->database = $database;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   * Page callback for Gutenberg block report.
   */
  public function report() {
    $header = [
      'block' => $this->t('Block Name'),
      'count' => $this->t('Usage Count'),
      'nodes' => $this->t('Used in Nodes'),
    ];

    $rows = [];

    // Get only body fields with Gutenberg format.
    $query = $this->database->select('node__body', 'b')
      ->fields('b', ['entity_id', 'body_value', 'body_format'])
      ->condition('body_format', 'gutenberg');
    $result = $query->execute();

    $block_usage = [];

    foreach ($result as $record) {
      if ($node = Node::load($record->entity_id)) {
        $node_link = $node->toLink()->toString();
      }
      else {
        $node_link = NULL;
      }

      preg_match_all('/<!--\s*wp:([a-zA-Z0-9\-\/_]+).*?-->/', $record->body_value, $matches);

      if (!empty($matches[1])) {
        foreach ($matches[1] as $block_name) {
          $block_usage[$block_name]['count'] = ($block_usage[$block_name]['count'] ?? 0) + 1;
          if ($node_link) {
            $block_usage[$block_name]['nodes'][] = $node_link;
          }
        }
      }
    }

    uasort($block_usage, function ($a, $b) {
      return ($b['count'] ?? 0) <=> ($a['count'] ?? 0);
    });

    foreach ($block_usage as $block_name => $info) {
      $unique_nodes = array_values(array_unique($info['nodes'] ?? []));
      $max_items = 10;
      $display_nodes = array_slice($unique_nodes, 0, $max_items);
      $items = [];
      foreach ($display_nodes as $link_markup) {
        $items[] = Markup::create($link_markup);
      }
      $remaining = ($info['count'] ?? 0) - count($display_nodes);
      if ($remaining > 0) {
        $more_url = Url::fromRoute('gutenberg_block_report.block_nodes', [], [
          'query' => ['block' => $block_name],
        ]);
        $more_text = $this->formatPlural($remaining, '… 1 more occurrence', '… @count more occurrences');
        $items[] = Link::fromTextAndUrl($more_text, $more_url)->toRenderable();
      }

      $display_name = $this->getBlockDisplayName($block_name);

      $rows[] = [
        'block' => [
          'data' => [
            '#type' => 'inline_template',
            '#template' => '<div class="block-display-name">{{ display }}</div><div class="block-machine-name"><code>[{{ machine }}]</code></div>',
            '#context' => [
              'display' => $display_name,
              'machine' => $block_name,
            ],
          ],
        ],
        'count' => $info['count'],
        'nodes' => [
          'data' => [
            '#theme' => 'item_list',
            '#items' => $items,
            '#attributes' => ['class' => ['block-report-node-list']],
          ],
        ],
      ];
    }

    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No Gutenberg blocks found.'),
    ];
  }

  /**
   * Detail page: list all nodes that use a given Gutenberg block.
   */
  public function blockNodes() {
    $request = \Drupal::request();
    $block_name = $request->query->get('block');

    if (empty($block_name)) {
      return [
        '#markup' => $this->t('No block specified.'),
      ];
    }

    $query = $this->database->select('node__body', 'b')
      ->fields('b', ['entity_id', 'body_value', 'body_format'])
      ->condition('body_format', 'gutenberg')
      ->condition('body_value', '%<!-- wp:' . $this->database->escapeLike($block_name) . '%', 'LIKE');
    $result = $query->execute();
    $node_counts = [];
    foreach ($result as $record) {
      if (preg_match_all('/<!--\s*wp:([a-zA-Z0-9\-\/_]+).*?-->/', $record->body_value, $matches)) {
        if (!empty($matches[1])) {
          $count = 0;
          foreach ($matches[1] as $m) {
            if ($m === $block_name) {
              $count++;
            }
          }
          if ($count > 0) {
            $node_counts[(int) $record->entity_id] = $count;
          }
        }
      }
    }
    ksort($node_counts, SORT_NUMERIC);
    $links = [];
    $total_occurrences = 0;
    foreach ($node_counts as $nid => $occurrences) {
      $total_occurrences += $occurrences;
      if ($node = Node::load($nid)) {
        $occ_text = (string) $this->formatPlural($occurrences, '1 occurrence', '@count occurrences');
        $links[] = Markup::create($node->toLink()->toString() . ' (' . $occ_text . ')');
      }
    }

    $summary = $this->t('Total: @occ occurrences across @nodes nodes.', [
      '@occ' => $total_occurrences,
      '@nodes' => count($node_counts),
    ]);

    $build = [
      '#type' => 'container',
      'title' => [
        '#markup' => '<h2>' . $this->t('Nodes using block: @block', ['@block' => $block_name]) . '</h2>',
      ],
      'summary' => [
        '#markup' => '<p>' . $summary . '</p>',
      ],
      'list' => [
        '#theme' => 'item_list',
        '#items' => $links,
        '#attributes' => ['class' => ['block-report-node-list-all']],
      ],
    ];

    return $build;
  }

  /**
   * Derive a human-friendly display name from a Gutenberg block machine name.
   *
   * Example: "drupalmedia/drupal-media-entity" => "Drupal Media Entity".
   */
  protected function getBlockDisplayName(string $block_name): string {
    $parts = explode('/', $block_name);
    $slug = end($parts) ?: $block_name;
    $slug = str_replace(['-', '_'], ' ', $slug);
    $slug = preg_replace('/\s+/', ' ', $slug);
    $slug = trim($slug);
    return ucwords($slug);
  }
}
