<?php

namespace Drupal\gutenberg_block_report\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Render\Markup;
use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Controller for Gutenberg Block Report.
 */
class BlockReportController extends ControllerBase {

  protected $database;

  public function __construct(Connection $database) {
    $this->database = $database;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   * Page callback for Gutenberg block report.
   */
  public function report() {
    $header = [
      'block' => $this->t('Block Name'),
      'count' => $this->t('Usage Count'),
      'nodes' => $this->t('Used in Nodes'),
    ];

    $rows = [];

    // Get only body fields with Gutenberg format.
    $query = $this->database->select('node__body', 'b')
      ->fields('b', ['entity_id', 'body_value'])
      ->condition('body_format', 'gutenberg');
    $result = $query->execute();

    $block_usage = [];

    foreach ($result as $record) {
      preg_match_all('/<!--\s*wp:([a-zA-Z0-9\-\/_.]+)[^>]*-->/', $record->body_value, $matches);

      if (!empty($matches[1])) {
        foreach ($matches[1] as $block_name) {
          $block_usage[$block_name]['count'] = ($block_usage[$block_name]['count'] ?? 0) + 1;
          $block_usage[$block_name]['node_ids'][] = (int) $record->entity_id;
        }
      }
    }

    uasort($block_usage, function ($a, $b) {
      return ($b['count'] ?? 0) <=> ($a['count'] ?? 0);
    });

    $node_storage = \Drupal::entityTypeManager()->getStorage('node');
    foreach ($block_usage as $block_name => $info) {
      $unique_node_ids = array_values(array_unique($info['node_ids'] ?? []));
      $max_items = 10;
      $display_node_ids = array_slice($unique_node_ids, 0, $max_items);

      $items = [];
      if (!empty($display_node_ids)) {
        /** @var \Drupal\node\NodeInterface[] $nodes */
        $nodes = $node_storage->loadMultiple($display_node_ids);
        foreach ($display_node_ids as $nid) {
          if (!empty($nodes[$nid])) {
            $items[] = Markup::create($nodes[$nid]->toLink()->toString());
          }
        }
      }

      $remaining = max(0, ($info['count'] ?? 0) - count($display_node_ids));
      if ($remaining > 0) {
        $more_url = Url::fromRoute('gutenberg_block_report.block_nodes', [], [
          'query' => ['block' => $block_name],
        ]);
        $more_text = $this->formatPlural($remaining, '… 1 more occurrence', '… @count more occurrences');
        $items[] = Link::fromTextAndUrl($more_text, $more_url)->toRenderable();
      }

      $display_name = $this->getBlockDisplayName($block_name);

      $rows[] = [
        'block' => [
          'data' => [
            '#type' => 'inline_template',
            '#template' => '<div class="block-display-name">{{ display }}</div><div class="block-machine-name"><code>[{{ machine }}]</code></div>',
            '#context' => [
              'display' => $display_name,
              'machine' => $block_name,
            ],
          ],
        ],
        'count' => $info['count'],
        'nodes' => [
          'data' => [
            '#theme' => 'item_list',
            '#items' => $items,
            '#attributes' => ['class' => ['block-report-node-list']],
          ],
        ],
      ];
    }

    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No Gutenberg blocks found.'),
      '#cache' => [
        'tags' => ['node_list'],
        'contexts' => ['user.permissions'],
        'max-age' => -1,
      ],
    ];
  }

  /**
   * Detail page: list all nodes that use a given Gutenberg block.
   */
  public function blockNodes() {
    $request = \Drupal::request();
    $block_name = $request->query->get('block');

    if (empty($block_name)) {
      return [
        '#markup' => $this->t('No block specified.'),
      ];
    }

    $query = $this->database->select('node__body', 'b')
      ->fields('b', ['entity_id', 'body_value'])
      ->condition('body_format', 'gutenberg')
      ->condition('body_value', '%<!-- wp:' . $this->database->escapeLike($block_name) . '%', 'LIKE');
    $result = $query->execute();
    $node_counts = [];
    foreach ($result as $record) {
      if (preg_match_all('/<!--\s*wp:([a-zA-Z0-9\-\/_.]+)[^>]*-->/', $record->body_value, $matches)) {
        if (!empty($matches[1])) {
          $count = 0;
          foreach ($matches[1] as $m) {
            if ($m === $block_name) {
              $count++;
            }
          }
          if ($count > 0) {
            $node_counts[(int) $record->entity_id] = $count;
          }
        }
      }
    }
    ksort($node_counts, SORT_NUMERIC);

    // Implement pagination to avoid rendering tens of thousands of links at once.
    $total_nodes = count($node_counts);
    $total_occurrences = array_sum($node_counts);
    $page = max(0, (int) $request->query->get('page'));
    $page_size = 100;
    $all_nids = array_keys($node_counts);
    $offset = $page * $page_size;
    $page_nids = array_slice($all_nids, $offset, $page_size, FALSE);
    $titles_map = [];
    if (!empty($page_nids)) {
      $title_query = $this->database->select('node_field_data', 'nfd')
        ->fields('nfd', ['nid', 'title'])
        ->condition('nfd.nid', $page_nids, 'IN');
      $title_result = $title_query->execute();
      foreach ($title_result as $row) {
        $titles_map[(int) $row->nid] = $row->title;
      }
    }

    $links = [];
    foreach ($page_nids as $nid) {
      $occurrences = $node_counts[$nid];
      $occ_text = (string) $this->formatPlural($occurrences, '1 occurrence', '@count occurrences');
      $title = $titles_map[$nid] ?? $this->t('Node @nid', ['@nid' => $nid]);
      $url = Url::fromRoute('entity.node.canonical', ['node' => $nid]);
      $links[] = Link::fromTextAndUrl($title . ' (' . $occ_text . ')', $url)->toRenderable();
    }

    $summary = $this->t('Total: @occ occurrences across @nodes nodes.', [
      '@occ' => $total_occurrences,
      '@nodes' => $total_nodes,
    ]);

    $build = [
      '#type' => 'container',
      'title' => [
        '#markup' => '<h2>' . $this->t('Nodes using block: @block (@machine)', ['@block' => $this->getBlockDisplayName($block_name), '@machine' => $block_name]) . '</h2>',
      ],
      'summary' => [
        '#markup' => '<p>' . $summary . '</p>',
      ],
      'list' => [
        '#theme' => 'item_list',
        '#items' => $links,
        '#attributes' => ['class' => ['block-report-node-list-all']],
      ],
      'pager' => [
        '#type' => 'pager',
        '#quantity' => 9,
      ],
      '#cache' => [
        'tags' => ['node_list'],
        'contexts' => ['url.query_args:block', 'url.query_args:page', 'user.permissions'],
        'max-age' => -1,
      ],
    ];

    $pager_manager = \Drupal::service('pager.manager');
    $pager = $pager_manager->createPager($total_nodes, $page_size);

    return $build;
  }

  /**
   * Derive a human-friendly display name from a Gutenberg block machine name.
   *
   * Example: "drupalmedia/drupal-media-entity" => "Drupal Media Entity".
   */
  protected function getBlockDisplayName(string $block_name): string {
    $parts = explode('/', $block_name);
    $slug = end($parts) ?: $block_name;
    $slug = str_replace(['-', '_'], ' ', $slug);
    $slug = preg_replace('/\s+/', ' ', $slug);
    $slug = trim($slug);
    return ucwords($slug);
  }
}
