<?php

namespace Drupal\h5peditor\H5PEditor;

use Drupal\h5peditor;
use Drupal\h5p\H5PDrupal\H5PDrupal;
use Drupal\Core\Url;

class H5PEditorUtilities {

  /**
   * Keeps track of our editor instance, saving valuable resources.
   *
   * @return \H5peditor
   */
  public static function getInstance() {
    static $instance;

    if (!$instance) {

      $core     = H5PDrupal::getInstance('core');
      $instance = new \H5peditor(
        $core,
        new H5PEditorDrupalStorage(),
        new H5PEditorDrupalAjax()
      );
    }

    return $instance;
  }

  /**
   * Get editor settings needed for JS front-end
   *
   * @return array Settings needed for view
   */
  public static function getEditorSettings() {
    $contentValidator = H5PDrupal::getInstance('contentvalidator');

    $settings = [
      'filesPath'          => base_path() . H5PDrupal::getRelativeH5PPath(),
      'fileIcon'           => [
        'path' => \Drupal::service('file_url_generator')->generate("assets://h5p/h5p-editor/images/binary-file.png", [
          'absolute' => TRUE,
          'language' => FALSE,
        ])->toString(TRUE)->getGeneratedUrl(),
        'width'  => 50,
        'height' => 50,
      ],
      'ajaxPath' => str_replace('%3A', ':', self::getAjaxPath()),
      'libraryPath' => \Drupal::service('file_url_generator')->generate("assets://h5p/h5p-editor/", [
          'absolute' => TRUE,
          'language' => FALSE,
        ])->toString(TRUE)->getGeneratedUrl() . base_path(),
      'copyrightSemantics' => $contentValidator->getCopyrightSemantics(),
      'metadataSemantics' => $contentValidator->getMetadataSemantics(),
      'assets'             => self::getEditorAssets(),
      'apiVersion'         => \H5PCore::$coreApi,
      'language' => \Drupal::languageManager()->getCurrentLanguage()->getId()
    ];

    return $settings;
  }

  /**
   * Get assets needed to display editor. These are fetched from core.
   *
   * @return array Js and css for showing the editor
   */
  private static function getEditorAssets() {
    $corePath = \Drupal::service('file_url_generator')->generate("assets://h5p/h5p-core/", [
      'absolute' => TRUE,
      'language' => FALSE,
    ])->toString(TRUE)->getGeneratedUrl() . base_path();
    $editorPath = \Drupal::service('file_url_generator')->generate("assets://h5p/h5p-editor/", [
      'absolute' => TRUE,
      'language' => FALSE,
    ])->toString(TRUE)->getGeneratedUrl() . base_path();

    $css = array_merge(
      self::getAssets(\H5PCore::$styles, $corePath),
      self::getAssets(\H5PEditor::$styles, $editorPath)
    );
    $js = array_merge(
      self::getAssets(\H5PCore::$scripts, $corePath),
      self::getAssets(\H5PEditor::$scripts, $editorPath, ['scripts/h5peditor-editor.js'])
    );
    $js[] = self::getTranslationFilePath();

    return ['css' => $css, 'js' => $js];
  }

  /**
   * Extracts assets from a collection of assets
   *
   * @param array $collection Collection of assets
   * @param string $prefix Prefix needed for constructing the file-path of the assets
   * @param null|array $exceptions Exceptions from the collection that should be skipped
   *
   * @return array Extracted assets from the source collection
   */
  private static function getAssets($collection, $prefix, $exceptions = NULL) {
    $assets      = [];
    $cacheBuster = self::getCacheBuster();

    foreach ($collection as $item) {
      // Skip exceptions
      if ($exceptions && in_array($item, $exceptions)) {
        continue;
      }
      $assets[] = "{$prefix}{$item}{$cacheBuster}";
    }
    return $assets;
  }

  /**
   * Get cache buster
   *
   * @return string A cache buster that may be applied to resources
   */
  private static function getCacheBuster() {
    $cache_buster = \Drupal::service('asset.query_string')->get();
    return $cache_buster ? "?{$cache_buster}" : '';
  }

  /**
   * Translation file path for the editor. Defaults to English if chosen
   * language is not available.
   *
   * @return string Path to translation file for editor
   */
  private static function getTranslationFilePath() {
    $language = \Drupal::languageManager()->getCurrentLanguage()->getId();

    $languageFolder = "assets://h5p/h5p-editor/language";
    $defaultLanguage = \Drupal::service('file_url_generator')->generate("{$languageFolder}/en.js", [
      'absolute' => TRUE,
      'language' => FALSE,
    ])->toString(TRUE)->getGeneratedUrl();
    $chosenLanguage = \Drupal::service('file_url_generator')->generate("{$languageFolder}/{$language}.js", [
      'absolute' => TRUE,
      'language' => FALSE,
    ])->toString(TRUE)->getGeneratedUrl();
    $cacheBuster = self::getCacheBuster();

    return (file_exists(DRUPAL_ROOT . $chosenLanguage) ? $chosenLanguage : $defaultLanguage) . $cacheBuster;
  }

  /**
   * Create URI for ajax the client may send to the server
   *
   * @return \Drupal\Core\GeneratedUrl|string Uri for AJAX
   */
  private static function getAjaxPath() {
    $securityToken = \H5PCore::createToken('editorajax');
    return Url::fromUri(
      "internal:/h5peditor/{$securityToken}/:contentId/"
    )->toString();
  }

  /**
   * Extract library information from library string
   *
   * @param string $library Library string with versioning, e.g. H5P.MultiChoice 1.9
   * @param string $property May be used to only extract certain information
   * about library. Available values are 'all', 'libraryId' and specific property
   *
   * @return int|bool|array One or more properties, or false if invalid.
   */
  public static function getLibraryProperty($library, $property = 'all') {
    $matches = [];
    preg_match_all('/(.+)\s(\d+)\.(\d+)$/', $library, $matches);
    if (count($matches) == 4) {
      $libraryData = [
        'name'         => $matches[1][0],
        'machineName'  => $matches[1][0],
        'majorVersion' => $matches[2][0],
        'minorVersion' => $matches[3][0],
      ];
      switch ($property) {
        case 'all':
          $libraryData['libraryId'] = self::getLibraryId($libraryData);
          return $libraryData;
        case 'libraryId':
          $libraryId = self::getLibraryId($libraryData);
          return $libraryId;
        default:
          return $libraryData[$property];
      }
    }
    else {
      return FALSE;
    }
  }

  /**
   * Library ID from unique library data
   *
   * @param array $libraryData Library data which must contain
   * 'machineName', 'majorVersion' and 'minorVersion'
   *
   * @return null|int Library id
   */
  private static function getLibraryId($libraryData) {
    $select = \Drupal::database()->select('h5p_libraries');
    $select->fields('h5p_libraries', array('library_id'))
           ->condition('machine_name', $libraryData['machineName'])
           ->condition('major_version', $libraryData['majorVersion'])
           ->condition('minor_version', $libraryData['minorVersion']);
    return $select->execute()->fetchField();
  }
}
