<?php

namespace Drupal\hal_publications\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityPublishedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Defines the HAL Author entity.
 *
 * @ContentEntityType(
 *   id = "hal_author",
 *   label = @Translation("HAL Author"),
 *   label_collection = @Translation("HAL Authors"),
 *   label_singular = @Translation("HAL author"),
 *   label_plural = @Translation("HAL authors"),
 *   label_count = @PluralTranslation(
 *     singular = "@count HAL author",
 *     plural = "@count HAL authors"
 *   ),
 *   base_table = "hal_author",
 *   data_table = "hal_author_field_data",
 *   revision_table = "hal_author_revision",
 *   revision_data_table = "hal_author_field_revision",
 *   translatable = TRUE,
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "title",
 *     "uuid" = "uuid",
 *     "langcode" = "langcode",
 *     "revision" = "revision_id",
 *     "published" = "status",
 *   },
 *   handlers = {
 *     "storage" = "Drupal\hal_publications\Entity\HalAuthorStorage",
 *     "storage_schema" = "Drupal\hal_publications\Entity\HalAuthorStorageSchema",
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\hal_publications\Entity\HalAuthorListBuilder",
 *     "access" = "Drupal\hal_publications\Entity\HalAuthorAccessControlHandler",
 *     "form" = {
 *       "default" = "Drupal\hal_publications\Form\HalAuthorForm",
 *       "add" = "Drupal\hal_publications\Form\HalAuthorForm",
 *       "edit" = "Drupal\hal_publications\Form\HalAuthorForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   admin_permission = "administer hal authors",
 *   links = {
 *     "add-form" = "/admin/content/hal-author/add",
 *     "edit-form" = "/admin/content/hal-author/{hal_author}/edit",
 *     "delete-form" = "/admin/content/hal-author/{hal_author}/delete",
 *     "collection" = "/admin/content/hal-author",
 *   },
 *   field_ui_base_route = "entity.hal_author.collection",
 * )
 */
class HalAuthor extends ContentEntityBase implements HalAuthorInterface, EntityPublishedInterface, EntityChangedInterface {

  use StringTranslationTrait;
  use EntityPublishedTrait;
  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    // Title field (author name, used as entity label).
    $fields['title'] = BaseFieldDefinition::create('string')
      ->setLabel(\Drupal::translation()->translate('Name'))
      ->setDescription(\Drupal::translation()->translate('The name of the HAL author (e.g., "Last, First" or full name). Used for display in admin interface.'))
      ->setRequired(TRUE)
      ->setTranslatable(TRUE)
      ->setRevisionable(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // HAL identifier field (required, used for API queries).
    $fields['hal_id'] = BaseFieldDefinition::create('string')
      ->setLabel(\Drupal::translation()->translate('HAL Identifier'))
      ->setDescription(\Drupal::translation()->translate('The HAL identifier for this author (e.g., "firstname-lastname"). This is not a number.'))
      ->setRequired(TRUE)
      ->setTranslatable(TRUE)
      ->setRevisionable(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Published status field.
    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(\Drupal::translation()->translate('Published'))
      ->setDescription(\Drupal::translation()->translate('A boolean indicating whether the HAL Author is published.'))
      ->setTranslatable(TRUE)
      ->setRevisionable(TRUE)
      ->setDefaultValue(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => TRUE,
        ],
        'weight' => 100,
      ])
      ->setDisplayConfigurable('form', TRUE);

    // Created timestamp.
    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(\Drupal::translation()->translate('Created'))
      ->setDescription(\Drupal::translation()->translate('The time that the HAL Author was created.'))
      ->setTranslatable(TRUE)
      ->setRevisionable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'timestamp',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE);

    // Changed timestamp.
    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(\Drupal::translation()->translate('Changed'))
      ->setDescription(\Drupal::translation()->translate('The time that the HAL Author was last edited.'))
      ->setTranslatable(TRUE)
      ->setRevisionable(TRUE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getHalId() {
    return $this->get('hal_id')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setHalId($hal_id) {
    $this->set('hal_id', $hal_id);
    return $this;
  }

}
