<?php

namespace Drupal\hal_publications\Form;

use Drupal\hal_publications\Service\HalPublicationsApiService;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\StringTranslation\TranslationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for HAL Publications settings.
 */
class HalPublicationsConfigForm extends ConfigFormBase {

  /**
   * The HAL API service.
   *
   * @var \Drupal\hal_publications\Service\HalPublicationsApiService
   */
  protected $halApi;

  /**
   * Constructs the form.
   */
  public function __construct(TranslationInterface $string_translation, HalPublicationsApiService $hal_api) {
    $this->stringTranslation = $string_translation;
    $this->halApi = $hal_api;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('string_translation'),
      $container->get('hal_publications.api')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'hal_publications_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['hal_publications.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('hal_publications.settings');

    $form['portals_collections'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Portal and Collection management'),
      '#description' => $this->t('Manage portals and collections through this interface. These options will be available in block configuration. <strong>Note:</strong> Portals are lowercase (e.g., "tel"), collections are uppercase (e.g., "FRANCE-GRILLES").'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    $portals = $config->get('portals') ?: [];
    $portals_count = count($portals);
    $portal_description = $this->t('Enter portals, one per line, in the format: <code>portal_id|Portal Label</code>. Example: <code>tel|TEL Portal</code> or <code>shs|HAL SHS</code>. Portal IDs must be lowercase. <a href="https://api.archives-ouvertes.fr/ref/instance" target="_blank">View all available portals</a>.');
    if ($portals_count > 0) {
      $portal_description .= ' ' . $this->t('Currently @count portal(s) configured.', ['@count' => $portals_count]);
    }
    else {
      $portal_description .= ' ' . $this->t('<strong>No portals configured yet.</strong>');
    }
    $form['portals_collections']['portals_list'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Portals'),
      '#description' => Markup::create($portal_description),
      '#default_value' => $this->formatPortalsForTextarea($portals),
      '#rows' => 5,
      '#required' => FALSE,
    ];

    $collections = $config->get('collections') ?: [];
    $collections_count = count($collections);
    $form['portals_collections']['collections_list'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Collections'),
      '#description' => $this->t('Enter collections, one per line, in the format: <code>collection_id|Collection Label</code>. Example: <code>FRANCE-GRILLES|France Grilles</code>. Collection IDs must be uppercase.') .
      ($collections_count > 0 ? ' ' . $this->t('Currently @count collection(s) configured.', ['@count' => $collections_count]) : ' ' . $this->t('<strong>No collections configured yet.</strong>')),
      '#default_value' => $this->formatCollectionsForTextarea($collections),
      '#rows' => 5,
      '#required' => FALSE,
    ];

    $form['query_defaults'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Query defaults'),
      '#description' => $this->t('Default settings for HAL API queries.'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    $form['query_defaults']['api_timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('API timeout'),
      '#description' => $this->t('API request timeout in seconds.'),
      '#default_value' => $config->get('api_timeout') ?? 30,
      '#min' => 1,
      '#max' => 300,
      '#required' => TRUE,
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced settings'),
      '#open' => FALSE,
    ];

    $form['advanced']['disable_ssl_verification'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable SSL certificate verification'),
      '#description' => $this->t('<strong>WARNING:</strong> Only enable this for development/testing. Disabling SSL verification is a security risk and should <strong>NEVER</strong> be used in production environments.'),
      '#default_value' => (bool) $config->get('disable_ssl_verification'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('hal_publications.settings');

    $existing_config = $config->get();
    $portals_collections_values = $form_state->getValue('portals_collections');
    $portals_list_direct = $form_state->getValue(['portals_collections', 'portals_list']);
    $collections_list_direct = $form_state->getValue(['portals_collections', 'collections_list']);
    $portals = [];
    $collections = [];

    $portals_text = NULL;
    $portals_text_submitted = FALSE;
    if ($portals_collections_values !== NULL && is_array($portals_collections_values) && isset($portals_collections_values['portals_list'])) {
      $portals_text = trim($portals_collections_values['portals_list'] ?? '');
      $portals_text_submitted = TRUE;
    }
    elseif ($portals_list_direct !== NULL) {
      $portals_text = trim($portals_list_direct);
      $portals_text_submitted = TRUE;
    }
    else {
      $user_input = $form_state->getUserInput();
      if (isset($user_input['portals_collections']['portals_list'])) {
        $portals_text = trim($user_input['portals_collections']['portals_list'] ?? '');
        $portals_text_submitted = TRUE;
      }
      elseif (isset($user_input['portals_list'])) {
        $portals_text = trim($user_input['portals_list'] ?? '');
        $portals_text_submitted = TRUE;
      }
    }

    if ($portals_text !== NULL && $portals_text !== '') {
      $lines = explode("\n", $portals_text);
      foreach ($lines as $line) {
        $line = trim($line);
        if (empty($line)) {
          continue;
        }
        $parts = explode('|', $line, 2);
        if (count($parts) === 2) {
          $portal_id = strtolower(trim($parts[0]));
          $portal_label = trim($parts[1]);
          if (!empty($portal_id) && !empty($portal_label)) {
            $portals[] = [
              'id' => $portal_id,
              'label' => $portal_label,
            ];
          }
        }
      }
    }

    $collections_text = NULL;
    $collections_text_submitted = FALSE;
    if ($portals_collections_values !== NULL && is_array($portals_collections_values) && isset($portals_collections_values['collections_list'])) {
      $collections_text = trim($portals_collections_values['collections_list'] ?? '');
      $collections_text_submitted = TRUE;
    }
    elseif ($collections_list_direct !== NULL) {
      $collections_text = trim($collections_list_direct);
      $collections_text_submitted = TRUE;
    }
    else {
      $user_input = $form_state->getUserInput();
      if (isset($user_input['portals_collections']['collections_list'])) {
        $collections_text = trim($user_input['portals_collections']['collections_list'] ?? '');
        $collections_text_submitted = TRUE;
      }
      elseif (isset($user_input['collections_list'])) {
        $collections_text = trim($user_input['collections_list'] ?? '');
        $collections_text_submitted = TRUE;
      }
    }

    if ($collections_text !== NULL && $collections_text !== '') {
      $lines = explode("\n", $collections_text);
      foreach ($lines as $line) {
        $line = trim($line);
        if (empty($line)) {
          continue;
        }
        $parts = explode('|', $line, 2);
        if (count($parts) === 2) {
          $collection_id = strtoupper(trim($parts[0]));
          $collection_label = trim($parts[1]);
          if (!empty($collection_id) && !empty($collection_label)) {
            $collections[] = [
              'id' => $collection_id,
              'label' => $collection_label,
            ];
          }
        }
      }
    }

    if ($portals_text_submitted) {
      $config->set('portals', $portals);
    }
    elseif (isset($existing_config['portals']) && is_array($existing_config['portals'])) {
      $config->set('portals', $existing_config['portals']);
    }
    else {
      $config->set('portals', []);
    }

    if ($collections_text_submitted) {
      $config->set('collections', $collections);
    }
    elseif (isset($existing_config['collections']) && is_array($existing_config['collections'])) {
      $config->set('collections', $existing_config['collections']);
    }
    else {
      $config->set('collections', []);
    }

    $disable_ssl_verification = $form_state->getValue('disable_ssl_verification');
    if ($disable_ssl_verification !== NULL) {
      $config->set('disable_ssl_verification', (bool) $disable_ssl_verification);
    }

    $query_defaults = $form_state->getValue('query_defaults');
    if ($query_defaults !== NULL && is_array($query_defaults)) {
      if (isset($query_defaults['api_timeout'])) {
        $config->set('api_timeout', (int) $query_defaults['api_timeout']);
      }
    }

    $config->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Formats portals array for textarea display.
   *
   * @param array $portals
   *   Array of portals with 'id' and 'label' keys.
   *
   * @return string
   *   Formatted string for textarea.
   */
  private function formatPortalsForTextarea(array $portals): string {
    $lines = [];
    foreach ($portals as $portal) {
      if (!empty($portal['id']) && !empty($portal['label'])) {
        $lines[] = $portal['id'] . '|' . $portal['label'];
      }
    }
    return implode("\n", $lines);
  }

  /**
   * Formats collections array for textarea display.
   *
   * @param array $collections
   *   Array of collections with 'id' and 'label' keys.
   *
   * @return string
   *   Formatted string for textarea.
   */
  private function formatCollectionsForTextarea(array $collections): string {
    $lines = [];
    foreach ($collections as $collection) {
      if (!empty($collection['id']) && !empty($collection['label'])) {
        $lines[] = $collection['id'] . '|' . $collection['label'];
      }
    }
    return implode("\n", $lines);
  }

}
