<?php

namespace Drupal\hal_publications\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * {@inheritdoc}
 */
class HalPublicationsFiltersForm extends FormBase {
  use StringTranslationTrait;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $currentRouteMatch;

  /**
   * Constructs a HalPublicationsFiltersForm object.
   *
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   */
  public function __construct(TranslationInterface $string_translation, RequestStack $request_stack, RouteMatchInterface $route_match) {
    $this->stringTranslation = $string_translation;
    $this->requestStack      = $request_stack;
    $this->currentRouteMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('string_translation'),
      $container->get('request_stack'),
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'hal_publications_filters_form';
  }

  /**
   * Filters form.
   *
   * @param array $form
   *   : Form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   : From state.
   * @param array $filters
   *   : Filters.
   * @param array $config
   *   : Block configuration.
   *
   * @return array
   *   : Form.
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?array $filters = NULL, ?array $config = NULL) {
    // Ensure filters and config have proper defaults.
    $filters = $filters ?? ['author' => [], 'year' => []];
    $config = $config ?? [
      'multiple_authors' => [1 => 0],
      'multiple_years' => [1 => 0],
    ];

    // Ensure nested arrays exist.
    if (!isset($filters['author']) || !is_array($filters['author'])) {
      $filters['author'] = [];
    }
    if (!isset($filters['year']) || !is_array($filters['year'])) {
      $filters['year'] = [];
    }
    if (!isset($config['multiple_authors'])) {
      $config['multiple_authors'] = [1 => 0];
    }
    if (!isset($config['multiple_years'])) {
      $config['multiple_years'] = [1 => 0];
    }

    // Store config in form array for submit handler.
    $multiple_authors = !empty($config['multiple_authors'][1]) && $config['multiple_authors'][1] == "1";
    $multiple_years = !empty($config['multiple_years'][1]) && $config['multiple_years'][1] == "1";

    $form['#multiple_authors'] = $multiple_authors;
    $form['#multiple_years'] = $multiple_years;

    $term = $this->requestStack->getCurrentRequest()->query->get('term');
    $author = NULL;
    $year = NULL;

    if ($this->requestStack->getCurrentRequest()->query->get('author')) {
      $author = explode('+', $this->requestStack->getCurrentRequest()->query->get('author'));
    }
    if ($this->requestStack->getCurrentRequest()->query->get('year')) {
      $year = explode('+', $this->requestStack->getCurrentRequest()->query->get('year'));
    }

    // Text filter.
    $form['term'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Word(s) present in title, abstract, etc.'),
      '#default_value' => $term ?? '',
    ];

    // Authors filter.
    $form['author'] = [
      '#type'          => 'select',
      '#id'            => 'edit-author',
      '#multiple'      => $multiple_authors,
      '#title'         => $this->t('Author(s)'),
      '#default_value' => $author ?? "",
      '#options'       => $filters['author'],
      '#attributes'    => [
        'class' => ['chosen-enable'],
      ],
      '#empty_option'  => $multiple_authors ? NULL : $this->t('- Select -'),
    ];

    // Years filter.
    $form['year'] = [
      '#type'          => 'select',
      '#id'            => 'edit-year',
      '#multiple'      => $multiple_years,
      '#title'         => $this->t('Year(s)'),
      '#default_value' => $year ?? "",
      '#options'       => $filters['year'],
      '#attributes'    => [
        'class' => ['chosen-enable'],
      ],
      '#empty_option'  => $multiple_years ? NULL : $this->t('- Select -'),
    ];

    // Filter button.
    $form['submit'] = [
      '#type'  => 'submit',
      '#value' => $this->t('Search'),
    ];

    // Reset button is only created if filters have already been chosen.
    if ($this->requestStack->getCurrentRequest()->query->get('term') || $this->requestStack->getCurrentRequest()->query->get('author') || $this->requestStack->getCurrentRequest()->query->get('year')) {
      // Reset button.
      $form['reset'] = [
        '#type'   => 'submit',
        '#value'  => $this->t('Reset'),
        '#submit' => ['::halFiltersReset'],
        '#limit_validation_errors' => [],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Get multiple flags from form array (stored in buildForm).
    $multiple_authors = !empty($form['#multiple_authors']) ? 1 : 0;
    $multiple_years = !empty($form['#multiple_years']) ? 1 : 0;

    $options = [];

    if ($form_state->getValue('term')) {
      $term = $form_state->getValue('term');
    }

    if ($form_state->getValue('author')) {
      $authors = "";
      if ($multiple_authors == 0) {
        $authors = $form_state->getValue('author');
      }
      else {
        $author_values = $form_state->getValue('author');
        if (is_array($author_values) && !empty($author_values)) {
          $last_author = end($author_values);
          foreach ($author_values as $author) {
            if ($last_author == $author) {
              $authors .= $author;
            }
            else {
              $authors .= $author . "+";
            }
          }
        }
      }
    }

    if ($form_state->getValue('year')) {
      $years = "";
      if ($multiple_years == "0") {
        $years = $form_state->getValue('year');
      }
      else {
        $year_values = $form_state->getValue('year');
        if (is_array($year_values) && !empty($year_values)) {
          $last_year = end($year_values);
          foreach ($year_values as $year) {
            if ($last_year == $year) {
              $years .= $year;
            }
            else {
              $years .= $year . "+";
            }
          }
        }
      }
    }

    if (isset($term)) {
      $options["query"]["term"] = $term;
    }

    if (isset($authors) && $authors !== "") {
      $options["query"]["author"] = $authors;
    }

    if (isset($years) && $years !== "") {
      $options["query"]["year"] = $years;
    }

    $node = $this->currentRouteMatch->getParameter('node');

    if ($node instanceof NodeInterface) {
      $form_state->setRedirect($this->currentRouteMatch->getRouteName(), ['node' => $node->id()], $options);
    }
    else {
      $form_state->setRedirect($this->currentRouteMatch->getRouteName(), [], $options);
    }
  }

  /**
   * Resets all filters.
   *
   * @param array $form
   *   : Form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   : From state.
   */
  public function halFiltersReset(array $form, FormStateInterface &$form_state) {
    $node = $this->currentRouteMatch->getParameter('node');

    if ($node instanceof NodeInterface) {
      $form_state->setRedirect($this->currentRouteMatch->getRouteName(), ['node' => $node->id()]);
    }
    else {
      $form_state->setRedirect($this->currentRouteMatch->getRouteName());
    }
  }

}
