<?php

declare(strict_types=1);

namespace Drupal\hal_publications\Hook;

use Drupal\hal_publications\Service\HalPublicationsApiService;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Hook implementations for hal_publications module.
 */
final class HalPublicationsHooks {

  use StringTranslationTrait;

  /**
   * The HAL API service.
   *
   * @var \Drupal\hal_publications\Service\HalPublicationsApiService
   */
  protected $halApi;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $currentRouteMatch;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs HalPublicationsHooks.
   *
   * @param \Drupal\hal_publications\Service\HalPublicationsApiService $hal_api
   *   The HAL API service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Routing\RouteMatchInterface $current_route_match
   *   The current route match service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(HalPublicationsApiService $hal_api, ConfigFactoryInterface $config_factory, RouteMatchInterface $current_route_match, EntityFieldManagerInterface $entity_field_manager, LoggerChannelFactoryInterface $logger_factory, EntityTypeManagerInterface $entity_type_manager) {
    $this->halApi = $hal_api;
    $this->configFactory = $config_factory;
    $this->currentRouteMatch = $current_route_match;
    $this->entityFieldManager = $entity_field_manager;
    $this->loggerFactory = $logger_factory;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Creates an instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('hal_publications.api'),
      $container->get('config.factory'),
      $container->get('current_route_match'),
      $container->get('entity_field.manager'),
      $container->get('logger.factory'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme(): array {
    return [
      'vancouver' => [
        'template' => 'vancouver',
        'variables' => [
          'docs'    => NULL,
          'display' => NULL,
          'config'  => NULL,
          'filters' => NULL,
          'author'  => NULL,
          'pagination' => NULL,
        ],
      ],
      'apa' => [
        'template' => 'apa',
        'variables' => [
          'docs'    => NULL,
          'display' => NULL,
          'config'  => NULL,
          'filters' => NULL,
          'author'  => NULL,
          'pagination' => NULL,
        ],
      ],
      'mla' => [
        'template' => 'mla',
        'variables' => [
          'docs'    => NULL,
          'display' => NULL,
          'config'  => NULL,
          'filters' => NULL,
          'author'  => NULL,
          'pagination' => NULL,
        ],
      ],
      'harvard' => [
        'template' => 'harvard',
        'variables' => [
          'docs'    => NULL,
          'display' => NULL,
          'config'  => NULL,
          'filters' => NULL,
          'author'  => NULL,
          'pagination' => NULL,
        ],
      ],
    ];
  }

  /**
   * Implements hook_preprocess_HOOK() for vancouver template.
   */
  #[Hook('preprocess_vancouver')]
  public function preprocessVancouver(array &$variables): void {
    if (isset($variables['docs']) && is_array($variables['docs'])) {
      foreach ($variables['docs'] as &$doc) {
        $doc += [
          'halId_s' => '',
          'authLastNameFirstName_s' => [],
          'authLastName_s' => [],
          'authFirstName_s' => [],
          'title_s' => [],
          'journalTitle_s' => [],
          'producedDateY_i' => '',
          'volume_s' => '',
          'issue_s' => [],
          'page_s' => '',
          'publisherLink_s' => [],
          'files_s' => [],
          'journalSherpaCondition_s' => '',
          'uri_s' => '',
          'label_xml' => '',
          'today_unix' => time(),
          'date_unix' => '',
          'embargo' => '',
          'doiId_s' => '',
        ];

        if (!is_array($doc['authLastNameFirstName_s'])) {
          $doc['authLastNameFirstName_s'] = [];
        }
        if (!is_array($doc['authLastName_s'])) {
          $doc['authLastName_s'] = [];
        }
        if (!is_array($doc['authFirstName_s'])) {
          $doc['authFirstName_s'] = [];
        }
        if (!is_array($doc['title_s'])) {
          $doc['title_s'] = [$doc['title_s']];
        }
        if (!is_array($doc['issue_s'])) {
          $doc['issue_s'] = [$doc['issue_s']];
        }
        if (!is_array($doc['publisherLink_s'])) {
          $doc['publisherLink_s'] = [$doc['publisherLink_s']];
        }
        if (!is_array($doc['files_s'])) {
          $doc['files_s'] = [$doc['files_s']];
        }
      }
      unset($doc);
    }

    if (!isset($variables['config']) || !is_array($variables['config'])) {
      $variables['config'] = [];
    }
    $variables['config'] += [
      'filtres' => [],
      'show_filters' => FALSE,
      'display_mode' => 'teaser',
    ];

    if (empty($variables['display'])) {
      $variables['display'] = 'teaser';
    }

    if (!isset($variables['pagination']) || !is_array($variables['pagination'])) {
      $variables['pagination'] = NULL;
    }

    $variables['has_docs'] = !empty($variables['docs']) && count($variables['docs']) > 0;
    $variables['docs_count'] = isset($variables['docs']) ? count($variables['docs']) : 0;
    $variables['show_filters'] = isset($variables['config']['filtres'][1]) && $variables['config']['filtres'][1] == 1;
  }

  /**
   * Implements hook_theme_suggestions_HOOK_alter() for vancouver.
   */
  #[Hook('theme_suggestions_vancouver_alter')]
  public function themeSuggestionsVancouverAlter(array &$suggestions, array $variables): void {
    if (!empty($variables['display'])) {
      $suggestions[] = 'vancouver__' . $variables['display'];
    }

    if (isset($variables['config']['block_id'])) {
      $suggestions[] = 'vancouver__block__' . $variables['config']['block_id'];
    }
  }

  /**
   * Implements hook_preprocess_HOOK() for apa template.
   */
  #[Hook('preprocess_apa')]
  public function preprocessApa(array &$variables): void {
    $this->preprocessVancouver($variables);
  }

  /**
   * Implements hook_preprocess_HOOK() for mla template.
   */
  #[Hook('preprocess_mla')]
  public function preprocessMla(array &$variables): void {
    $this->preprocessVancouver($variables);
  }

  /**
   * Implements hook_preprocess_HOOK() for harvard template.
   */
  #[Hook('preprocess_harvard')]
  public function preprocessHarvard(array &$variables): void {
    $this->preprocessVancouver($variables);
  }

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help(string $route_name, RouteMatchInterface $route_match): string {
    switch ($route_name) {
      case 'help.page.hal_publications':
        $output = '';
        $output .= '<p>' . $this->t("Drupal 10 / 11 module that connects to Archive Ouverte API to display publications.") . '</p>';
        $output .=
          '<p>
          <h2>' . $this->t('Requirements') . '</h2>
          <ul>
            <li>' . $this->t("Create HAL Author entities with the following fields:") . '
              <ul>
                <li>
                  ' . $this->t("HAL ID (required).") . '
                </li>
                <li>
                  ' . $this->t("First name (required).") . '
                </li>
                <li>
                  ' . $this->t("Last name (required).") . '
                </li>
                <li>
                  ' . $this->t("Title (full name in citation format: Last, First, required).") . '
                </li>
              </ul>
            </li>
            <li>
              ' . $this->t("Chosen Drupal module") . ' -
              <a href="https://www.drupal.org/project/chosen">
                https://www.drupal.org/project/chosen
              </a>.
            </li>
            <li>
              Download the ' . $this->t("Chosen") . ' -
              <a href="https://harvesthq.github.io/chosen/
              </a>. jQuery plugin:
                <ul>
                  <li>Via Drush: drush chosenplugin</li>
                  <li>Or extract the plugin under libraries/chosen.</li>
                </ul>
            </li>
          </ul>
        </p>
        <p>
          <h2>' . $this->t("Functionalities") . '</h2>
          <ul>
            <li>
              ' . $this->t("Choose publications source from multiple publications portals.") . '
            </li>
            <li>
              ' . $this->t("Choose whether to add an author page publications tab or not.") . '
            </li>
            <li>
              ' . $this->t("Display publications published by HAL authors.") .
          '
            </li>
            <li>
              ' . $this->t("Filter by keywords.") . '
            </li>
            <li>
              ' . $this->t("Filter by publication author(s).") . '
            </li>
            <li>
              ' . $this->t("Filter by publication year(s).") . '
            </li>
            <li>
              ' . $this->t("Display publications following the Vancouver format:") . '
              <a href="https://www.zotero.org/styles?q=id%3Avancouver-brackets-no-et-al">
                https://www.zotero.org/styles?q=id%3Avancouver-brackets-no-et-al
              </a>
              .
            </li>
            <li>
              ' . $this->t("Export a single or multiple publications in BibTex.") .
          '
            </li>
          </ul>
        </p>
        <p>
          <h2>' . $this->t("Add new HAL block") . '</h2>
          <ul>
            <li>
            ' . $this->t("Create and configure a block in") . '
              <a href="/admin/structure/block">
                Structure – Blocks
              </a>.
            </li>
            <li>
              ' . $this->t("Choose to either display filters or not.") . '
            </li>
            <li>
              ' . $this->t("Choose between a simple or multiple authors select list.") . '
            </li>
            <li>
              ' . $this->t("Choose between a simple or multiple years select list.") . '
            </li>
            <li>
              ' . $this->t("Filter publications by authors and years.") . '
            </li>
          </ul>
        </p>
        <p>
          <h2>' . $this->t("Publications by author") . '</h2>
          <ul>
            <li>
              ' . $this->t('Check the "Show user publications" option in the module configuration form:')
          . '
               <a href="/admin/config/hal">
                /admin/config/hal
               </a>.
            </li>
            <li>
              ' . $this->t('Fill the "User publications page title" field with a template (supports tokens: @fullname, @firstname, @lastname, @fullname_reversed).') .
          '
            </li>
          </ul>
        </p>';
        return $output;
    }
    return '';
  }

}
