<?php

namespace Drupal\hal_publications\Plugin\Block;

use Drupal\hal_publications\Service\HalPublicationsApiService;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Form\EnforcedResponseException;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides an 'HAL Publications' Block.
 *
 * @Block(
 *   id = "hal_publications",
 *   admin_label = @Translation("HAL Publications"),
 *   category = @Translation("HAL Publications"),
 * )
 */
class HalPublicationsBlock extends BlockBase implements ContainerFactoryPluginInterface {
  use StringTranslationTrait;

  /**
   * The string translation.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The HAL API service.
   *
   * @var \Drupal\hal_publications\Service\HalPublicationsApiService
   */
  protected $halApi;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a HalPublicationsBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\hal_publications\Service\HalPublicationsApiService $hal_api
   *   The HAL API service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, TranslationInterface $string_translation, FormBuilderInterface $form_builder, ConfigFactoryInterface $config_factory, Connection $database, HalPublicationsApiService $hal_api, LoggerChannelFactoryInterface $logger_factory, RequestStack $request_stack) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->stringTranslation = $string_translation;
    $this->formBuilder       = $form_builder;
    $this->configFactory     = $config_factory;
    $this->database          = $database;
    $this->halApi            = $hal_api;
    $this->logger            = $logger_factory->get('hal_publications');
    $this->requestStack      = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('string_translation'),
      $container->get('form_builder'),
      $container->get('config.factory'),
      $container->get('database'),
      $container->get('hal_publications.api'),
      $container->get('logger.factory'),
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $config = $this->getConfiguration();

    $config += [
      'filtres'                 => [],
      'multiple_authors'        => [],
      'multiple_years'          => [],
      'display_mode'            => '',
      'selected_portal'         => '',
      'selected_collection'     => '',
      'pagination_enabled'      => FALSE,
      'pagination_items_per_page' => 20,
    ];

    $filters = $this->halApi->getFilters($config);
    $url = $this->halApi->generateUrl($config);
    $data = $this->halApi->request($url, TRUE);

    // Ensure we have valid data.
    if (!is_array($data) || !isset($data["response"]["docs"]) || !is_array($data["response"]["docs"])) {
      $data = ["response" => ["docs" => []]];
    }

    $theme = $config['display_mode'] ?? '';
    if (empty($theme)) {
      $theme = 'vancouver';
    }

    $filters_form = NULL;
    if (!empty($config['filtres'][1])) {
      try {
        $filters_form = $this->formBuilder->getForm('Drupal\hal_publications\Form\HalPublicationsFiltersForm', $filters, $config);
      }
      catch (EnforcedResponseException $e) {
        throw $e;
      }
      catch (\Throwable $e) {
        $this->logger->error('Filter form error: @type - @message at @file:@line', [
          '@type' => get_class($e),
          '@message' => $e->getMessage(),
          '@file' => $e->getFile(),
          '@line' => $e->getLine(),
        ]);
        $filters_form = NULL;
      }
    }

    if (!empty($data["response"]["docs"])) {
      foreach ($data["response"]["docs"] as &$doc) {
        if (isset($doc["label_xml"]) && is_string($doc["label_xml"])) {
          $xml = @simplexml_load_string($doc["label_xml"]);
          if ($xml !== FALSE) {
            $date = $xml->text->body->listBibl->biblFull->editionStmt->edition->date[4] ?? NULL;
            if ($date) {
              $embargo   = $this->t("The file is embargoed until") . " " . $date;
              $date_unix = strtotime((string) $date);
            }
          }
        }

        $doc["today_unix"] = time();
        $doc["date_unix"]  = $date_unix ?? "";
        $doc["embargo"]    = $embargo ?? "";
      }
      unset($doc);
    }

    $pagination = NULL;
    if ($this->halApi->isPaginationEnabled($config)) {
      $num_found = (int) ($data["response"]["numFound"] ?? 0);
      $items_per_page = $this->halApi->getPaginationItemsPerPage($config);
      $total_pages = (int) ceil($num_found / $items_per_page);

      $pagination = [
        'current_page' => 0,
        'total_pages' => $total_pages,
        'items_per_page' => $items_per_page,
        'total_items' => $num_found,
        'client_side' => TRUE,
      ];
    }

    $plugin_id = $this->getPluginId();
    $block_id = 'hal-publications-block-' . $plugin_id;

    $build = [
      '#type' => 'container',
      '#attributes' => [
        'id' => $block_id,
        'class' => ['hal-publications-block-wrapper'],
        'data-block-id' => $plugin_id,
        'data-hal-block-plugin-id' => $plugin_id,
      ],
      'content' => [
        '#theme'   => $theme,
        '#docs'    => $data["response"]["docs"] ?? [],
        '#display' => 'teaser',
        '#config'  => $config,
        '#filters' => $filters_form,
        '#pagination' => $pagination,
        '#block_id' => $block_id,
      ],
      '#attached' => [
        'library' => [
          'hal_publications/libs',
        ],
        'drupalSettings' => [
          'halPublications' => [
            'blockId' => $block_id,
            'ajaxEnabled' => $this->halApi->isPaginationEnabled($config),
            'pagination' => $pagination ? [
              'itemsPerPage' => $pagination['items_per_page'],
              'totalPages' => $pagination['total_pages'],
              'totalItems' => $pagination['total_items'],
            ] : NULL,
          ],
        ],
      ],
      '#cache' => [
        'max-age' => 0,
        'contexts' => ['url.query_args', 'url.path'],
        'tags' => ['hal_publications:block'],
      ],
    ];

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);
    $config = $this->getConfiguration();

    $config += [
      'filtres'                 => [],
      'multiple_authors'        => [],
      'multiple_years'          => [],
      'display_mode'            => '',
      'selected_portal'         => '',
      'selected_collection'     => '',
      'pagination_enabled'      => FALSE,
      'pagination_items_per_page' => 20,
    ];

    $form['filtres'] = [
      '#type'          => 'checkboxes',
      '#title'         => $this->t('Show filters'),
      '#default_value' => $config["filtres"] ?? [],
      '#options'       => [
        1 => $this->t("Show"),
      ],
    ];

    $form['multiple_authors'] = [
      '#type'          => 'checkboxes',
      '#title'         => $this->t('Multiple selection list for authors'),
      '#states'        => [
        'visible' => [
          ':input[name="filtres"]' => [
            'value' => 1,
          ],
        ],
      ],
      '#default_value' => $config["multiple_authors"] ?? [],
      '#options'       => [
        1 => $this->t("Yes"),
      ],
    ];

    $form['multiple_years'] = [
      '#type'          => 'checkboxes',
      '#title'         => $this->t('Multiple selection list for years'),
      '#states'        => [
        'visible' => [
          ':input[name="filtres"]' => [
            'value' => 1,
          ],
        ],
      ],
      '#default_value' => $config["multiple_years"] ?? [],
      '#options'       => [
        1 => $this->t("Yes"),
      ],
    ];

    $portal_options = $this->halApi->getPortalOptions();
    $collection_options = $this->halApi->getCollectionOptions();

    $portal_options = ['' => $this->t('- No portal -')] + $portal_options;
    $collection_options = ['' => $this->t('- No collection -')] + $collection_options;

    $form['selected_portal'] = [
      '#type'          => 'select',
      '#title'         => $this->t('Portal'),
      '#description'   => $this->t('Select a portal for this block. If not selected, publications from all portals will be searched.'),
      '#default_value' => $config['selected_portal'] ?? '',
      '#options'       => $portal_options,
      '#required'      => FALSE,
    ];

    $form['selected_collection'] = [
      '#type'          => 'select',
      '#title'         => $this->t('Collection'),
      '#description'   => $this->t('Select a collection for this block. If not selected, publications from all collections will be searched. <strong>Note:</strong> Collection takes precedence over portal if both are set.'),
      '#default_value' => $config['selected_collection'] ?? '',
      '#options'       => $collection_options,
      '#required'      => FALSE,
    ];

    $form['display'] = [
      '#type'  => 'fieldset',
      '#title' => $this->t('Display'),
    ];

    $form['display']['display_mode'] = [
      '#type'          => 'select',
      '#title'         => $this->t('Citation format'),
      '#description'   => $this->t('Select the citation format for displaying publications.'),
      "#options"       => [
        '' => $this->t('- Select -'),
        'vancouver' => $this->t('Vancouver'),
        'apa' => $this->t('APA (American Psychological Association)'),
        'mla' => $this->t('MLA (Modern Language Association)'),
        'harvard' => $this->t('Harvard'),
      ],
      '#default_value' => $config['display_mode'] ?? '',
      '#required'      => TRUE,
    ];

    $form['pagination'] = [
      '#type'  => 'fieldset',
      '#title' => $this->t('Pagination'),
    ];

    $form['pagination']['pagination_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable pagination'),
      '#description' => $this->t('Enable pagination for large result sets using the start parameter.'),
      '#default_value' => $config['pagination_enabled'] ?? FALSE,
    ];

    $form['pagination']['pagination_items_per_page'] = [
      '#type' => 'number',
      '#title' => $this->t('Items per page'),
      '#description' => $this->t('Number of items per page when pagination is enabled.'),
      '#default_value' => $config['pagination_items_per_page'] ?? 20,
      '#min' => 1,
      '#max' => 1000,
      '#states' => [
        'visible' => [
          ':input[name="settings[pagination][pagination_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['display']['sort'] = [
      '#type' => 'select',
      '#title' => $this->t('Sort order'),
      '#description' => $this->t('How to sort the publications list.'),
      '#options' => [
        'producedDate_tdate+desc' => $this->t('Date (newest first)'),
        'producedDate_tdate+asc' => $this->t('Date (oldest first)'),
        'authLastNameFirstName_s+asc' => $this->t('Author (A-Z)'),
        'title_s+asc' => $this->t('Title (A-Z)'),
      ],
      '#default_value' => $config['sort'] ?? 'producedDate_tdate+desc',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $this->configuration['display_mode']              = $form_state->getValue('display')["display_mode"];
    $this->configuration['filtres']                   = $form_state->getValue('filtres');
    $this->configuration['multiple_authors']          = $form_state->getValue('multiple_authors');
    $this->configuration['multiple_years']            = $form_state->getValue('multiple_years');
    $this->configuration['selected_portal']           = $form_state->getValue('selected_portal') ?? '';
    $this->configuration['selected_collection']       = $form_state->getValue('selected_collection') ?? '';
    $this->configuration['pagination_enabled']        = (bool) ($form_state->getValue('pagination')['pagination_enabled'] ?? FALSE);
    $this->configuration['pagination_items_per_page'] = (int) ($form_state->getValue('pagination')['pagination_items_per_page'] ?? 20);
    $this->configuration['sort']                      = $form_state->getValue('display')['sort'] ?? 'producedDate_tdate+desc';
  }

}
