/**
 * @file
 * HAL Publications JavaScript: Chosen selects for filters and client-side pagination.
 * Uses Drupal behaviors for AJAX compatibility.
 */
(function ($, Drupal, once, drupalSettings) {
  'use strict';

  /**
   * Attach behaviors for HAL Publications.
   */
  Drupal.behaviors.halPublications = {
    attach: function (context, settings) {
      // Apply Chosen to selects once.
      once('hal-chosen', 'select#edit-year, select#edit-author', context).forEach(function (element) {
        applyChosen($(element));
      });

      // Match widths of chosen wrappers after initialization.
      once('hal-chosen-width', '.hal-filters', context).forEach(function (element) {
        matchChosenWidths($(element));
      });

      // Handle client-side pagination.
      // Search in context first, then fallback to document if not found.
      var $wrappers = $('.hal-publications-block-wrapper', context);
      if ($wrappers.length === 0) {
        $wrappers = $('.hal-publications-block-wrapper');
      }
      
      $wrappers.each(function() {
        var element = this;
        var processed = once('hal-pagination', element);
        if (processed.length > 0) {
          initClientSidePagination($(element));
        }
      });
    }
  };

  /**
   * Initialize client-side pagination for a HAL Publications block.
   */
  function initClientSidePagination($blockWrapper) {
    var $pagination = $blockWrapper.find('.hal-publications-pagination');
    if (!$pagination.length) {
      return; // No pagination needed.
    }

    // Find all publication items.
    var $publications = $blockWrapper.find('.hal-publication');
    if ($publications.length === 0) {
      return;
    }

    // Get pagination config from drupalSettings (Drupal way).
    var paginationConfig = null;
    if (drupalSettings && drupalSettings.halPublications && drupalSettings.halPublications.pagination) {
      paginationConfig = drupalSettings.halPublications.pagination;
    }
    
    // Get config values - use settings first, then fallback to data attributes.
    var itemsPerPage = paginationConfig ? paginationConfig.itemsPerPage : (parseInt($pagination.data('items-per-page')) || 20);
    var totalItems = paginationConfig ? paginationConfig.totalItems : $publications.length;
    var totalPages = paginationConfig ? paginationConfig.totalPages : Math.ceil(totalItems / itemsPerPage);
    var currentPage = 0;

    // Store reference to publications array for pagination.
    var publicationsArray = $publications.toArray();

    /**
     * Show a specific page of publications.
     */
    function showPage(page) {
      currentPage = page;
      var start = page * itemsPerPage;
      var end = Math.min(start + itemsPerPage, totalItems);

      // Hide ALL publications first - use both methods to be absolutely sure.
      $publications.hide().css('display', 'none');

      // Show only the publications for this page.
      for (var i = start; i < end && i < publicationsArray.length; i++) {
        $(publicationsArray[i]).show().css('display', 'block');
      }

      // Update UI.
      updatePaginationUI(page);
      
      // Scroll to top of block smoothly.
      var $blockTop = $blockWrapper;
      var scrollTop = $blockTop.offset().top - 100; // 100px offset from top
      if (scrollTop > 0) {
        $('html, body').animate({
          scrollTop: scrollTop
        }, 300);
      }
    }

    /**
     * Update pagination UI to reflect current page.
     */
    function updatePaginationUI(page) {
      // Update info text.
      var $info = $pagination.find('.hal-publications-pagination__info');
      if ($info.length) {
        $info.text('Page ' + (page + 1) + ' of ' + totalPages + ' (' + totalItems + ' items)');
      }

      // Remove current class from all items.
      $pagination.find('.hal-publications-pagination__item').removeClass('hal-publications-pagination__item--current');
      
      // Convert current span back to link if it exists.
      $pagination.find('.hal-publications-pagination__current').each(function() {
        var $span = $(this);
        var pageText = $span.text().trim();
        var pageNum = parseInt(pageText) - 1;
        if (isNaN(pageNum)) {
          return; // Skip if not a number
        }
        var $li = $span.parent();
        var $newLink = $('<a>').attr('href', '#').addClass('hal-publications-pagination__link').attr('data-page', pageNum).text(pageText);
        $span.replaceWith($newLink);
        $li.removeClass('hal-publications-pagination__item--current');
        
        // Re-attach click handler.
        $newLink.on('click.halPagination', function(e) {
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();
          showPage(pageNum);
          return false;
        });
      });

      // Mark new current page - find the link with matching page number.
      $pagination.find('.hal-publications-pagination__link').each(function() {
        var $link = $(this);
        var linkPage = parseInt($link.data('page'));
        
        if (!isNaN(linkPage) && linkPage === page) {
          var $li = $link.parent();
          var linkText = $link.text();
          $li.addClass('hal-publications-pagination__item--current');
          $link.replaceWith($('<span>').addClass('hal-publications-pagination__current').attr('aria-current', 'page').text(linkText));
        }
      });

      // Update Previous/Next button visibility.
      $pagination.find('.hal-publications-pagination__item--previous').toggle(page > 0);
      $pagination.find('.hal-publications-pagination__item--next').toggle(page < totalPages - 1);
      
      // Re-attach Previous/Next button handlers after UI update.
      attachPreviousNextHandlers();
    }

    /**
     * Attach event handlers to Previous/Next buttons.
     */
    function attachPreviousNextHandlers() {
      // Handle Previous button.
      $pagination.find('.hal-publications-pagination__item--previous .hal-publications-pagination__link').off('click.halPagination').on('click.halPagination', function(e) {
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();
        if (currentPage > 0) {
          showPage(currentPage - 1);
        }
        return false;
      });

      // Handle Next button.
      $pagination.find('.hal-publications-pagination__item--next .hal-publications-pagination__link').off('click.halPagination').on('click.halPagination', function(e) {
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();
        if (currentPage < totalPages - 1) {
          showPage(currentPage + 1);
        }
        return false;
      });
    }

    // Wire up ALL pagination links (number buttons).
    $pagination.find('.hal-publications-pagination__link').each(function() {
      var $link = $(this);
      var pageData = $link.data('page');
      
      // Skip Previous/Next buttons (they have 'prev' or 'next' as data-page).
      if (pageData === 'prev' || pageData === 'next') {
        return;
      }
      
      var pageNum = parseInt(pageData);
      if (!isNaN(pageNum) && pageNum >= 0 && pageNum < totalPages) {
        $link.off('click.halPagination').on('click.halPagination', function(e) {
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();
          showPage(pageNum);
          return false;
        });
      }
    });

    // Attach Previous/Next handlers initially.
    attachPreviousNextHandlers();

    // Initialize: hide all, show first page.
    $publications.hide().css('display', 'none');
    showPage(0);
  }

  /**
   * Apply Chosen to select elements.
   */
  function applyChosen($select) {
    if (!$select.length) {
      return;
    }

    try {
      // Get translated placeholder text from Drupal.
      var placeholder = '';
      if ($select.attr('id') === 'edit-year') {
        placeholder = Drupal.t('Year');
      }
      else if ($select.attr('id') === 'edit-author') {
        placeholder = Drupal.t('Author');
      }
      
      // Use Drupal's Chosen integration if available.
      if (typeof Drupal.chosen !== 'undefined' && Drupal.chosen.attach) {
        Drupal.chosen.attach($select);
      }
      // Fallback to direct Chosen API if Drupal integration doesn't work.
      else if (typeof $.fn.chosen !== 'undefined') {
        var options = {};
        if ($select.prop('multiple')) {
          options.placeholder_text_multiple = placeholder || Drupal.t('Choose some options');
        }
        else {
          options.placeholder_text_single = placeholder || Drupal.t('Choose an option');
        }
        $select.chosen(options);
      }
    }
    catch (e) {
      console.warn('Chosen initialization failed:', e);
    }
  }

  /**
   * Match widths of Chosen select wrappers.
   */
  function matchChosenWidths($container) {
    var $selects = $container.find('select');
    if ($selects.length > 1) {
      var maxWidth = 0;
      $selects.each(function () {
        var $chosen = $(this).next('.chosen-container');
        if ($chosen.length) {
          var width = $chosen.outerWidth();
          if (width > maxWidth) {
            maxWidth = width;
          }
        }
      });
      if (maxWidth > 0) {
        $selects.each(function () {
          var $chosen = $(this).next('.chosen-container');
          if ($chosen.length) {
            $chosen.css('width', maxWidth + 'px');
          }
        });
      }
    }
  }

})(jQuery, Drupal, once, drupalSettings);
