<?php

namespace Drupal\hal_publications\Service;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\ClientInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service for HAL API interactions.
 */
class HalPublicationsApiService {

  use StringTranslationTrait;

  /**
   * Base URL for HAL API requests.
   */
  private const HAL_API_BASE_URL = 'https://api.archives-ouvertes.fr';

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The Module Extension List.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs HalPublicationsApiService.
   */
  public function __construct(
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    RequestStack $request_stack,
    ModuleExtensionList $module_extension_list,
    FileSystemInterface $file_system,
  ) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('hal_publications');
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->requestStack = $request_stack;
    $this->moduleExtensionList = $module_extension_list;
    $this->fileSystem = $file_system;
  }

  /**
   * Gets the required HAL API fields for citation formats.
   *
   * @return string
   *   Comma-separated list of field names for the 'fl' parameter.
   */
  public function getApiFields(): string {
    $required_fields = [
      'authLastNameFirstName_s',
      'authLastName_s',
      'authFirstName_s',
      'title_s',
      'journalTitle_s',
      'producedDateY_i',
      'volume_s',
      'issue_s',
      'page_s',
      'publisherLink_s',
      'halId_s',
      'uri_s',
      'files_s',
      'journalSherpaCondition_s',
      'label_xml',
      'doiId_s',
    ];

    return implode(',', $required_fields);
  }

  /**
   * Gets the sort option from block configuration.
   *
   * @param array $block_config
   *   Optional block configuration array.
   *
   * @return string
   *   Sort field and direction (e.g., 'producedDate_tdate+desc').
   */
  public function getSort(array $block_config = []): string {
    if (isset($block_config['sort']) && !empty($block_config['sort'])) {
      return $block_config['sort'];
    }
    return 'producedDate_tdate+desc';
  }

  /**
   * Gets the maximum number of results per request.
   *
   * @return int
   *   Number of results per request (HAL API maximum is 10000).
   */
  public function getDefaultRows(): int {
    return 10000;
  }

  /**
   * Gets the HAL API filter field name.
   *
   * @param string $filter_type
   *   Filter type: 'author', 'year', 'text'.
   *
   * @return string
   *   The HAL API field name for filtering.
   */
  public function getFilterFieldName(string $filter_type): string {
    $filter_fields = [
      'author' => 'authIdHal_s',
      'year' => 'producedDateY_i',
      'text' => 'text',
    ];
    return $filter_fields[$filter_type] ?? '';
  }

  /**
   * Checks if pagination is enabled.
   *
   * @param array $block_config
   *   Optional block configuration array.
   *
   * @return bool
   *   TRUE if pagination is enabled.
   */
  public function isPaginationEnabled(array $block_config = []): bool {
    // Check block config first, then fall back to global config.
    if (isset($block_config['pagination_enabled'])) {
      return (bool) $block_config['pagination_enabled'];
    }
    // Fallback: default to disabled.
    return FALSE;
  }

  /**
   * Gets items per page for pagination.
   *
   * @param array $block_config
   *   Optional block configuration array.
   *
   * @return int
   *   Number of items per page.
   */
  public function getPaginationItemsPerPage(array $block_config = []): int {
    // Check block config first, then fall back to default.
    if (isset($block_config['pagination_items_per_page'])) {
      return (int) $block_config['pagination_items_per_page'];
    }
    // Fallback: default to 20 items per page.
    return 20;
  }

  /**
   * Gets facet sort order.
   *
   * @return string
   *   Sort order: 'count' (most common first).
   */
  public function getFacetSort(): string {
    return 'count';
  }

  /**
   * Gets facet limit.
   *
   * @return int
   *   Maximum number of facet values (100).
   */
  public function getFacetLimit(): int {
    return 100;
  }

  /**
   * Gets the configured API timeout.
   *
   * @return int
   *   Timeout in seconds.
   */
  public function getApiTimeout(): int {
    $config = $this->configFactory->get('hal_publications.settings');
    return (int) ($config->get('api_timeout') ?: 0);
  }

  /**
   * Sends requests & returns the result.
   *
   * @param string $url
   *   Request URL.
   * @param bool $decode
   *   Whether to JSON decode the response.
   *
   * @return string|array
   *   BibTex formatted string or decoded JSON array.
   */
  public function request(string $url, bool $decode = FALSE): array|string {
    $config = $this->configFactory->get('hal_publications.settings');

    $options = [
      'timeout' => $this->getApiTimeout(),
      'headers' => [
        'Accept' => 'application/json',
      ],
    ];

    if ($config->get('disable_ssl_verification')) {
      $options['verify'] = FALSE;
      $this->logger->warning('SSL verification is disabled. This is a security risk and should only be used for development.');
    }

    try {
      $result = $this->httpClient->request('GET', $url, $options);
      $body = $result->getBody()->getContents();

      if ($decode === TRUE) {
        $decoded = Json::decode($body);

        if (!is_array($decoded) || !isset($decoded['response'])) {
          $this->logger->error('HAL API returned invalid response structure. Body length: @len', [
            '@len' => strlen($body),
            '@body_start' => substr($body, 0, 200),
          ]);
          return ['response' => ['numFound' => 0, 'docs' => []]];
        }

        $num_found = $decoded['response']['numFound'] ?? 0;
        if ($num_found == 0) {
          $this->logger->warning('HAL API returned 0 results. Check URL and HAL IDs.');
        }
        return $decoded;
      }
      return $body;
    }
    catch (\Exception $e) {
      $this->logger->error('HAL API request failed: @message', [
        '@message' => $e->getMessage(),
        '@url' => $url,
      ]);

      return $decode ? ['response' => ['numFound' => 0, 'docs' => []]] : '';
    }
  }

  /**
   * Generates URL for API requests.
   *
   * @param array $config
   *   Configuration array.
   * @param bool $is_rows
   *   TRUE: URL for number of results only.
   *
   * @return string
   *   URL string.
   */
  public function generateUrl(array $config, bool $is_rows = FALSE): string {
    $base_url = rtrim(self::HAL_API_BASE_URL, '/');

    $selected_portal = trim($config['selected_portal'] ?? '');
    $selected_collection = trim($config['selected_collection'] ?? '');

    $url = $base_url . '/search/';
    if ($selected_collection) {
      $selected_collection = strtoupper($selected_collection);
      $url .= $selected_collection . '/';
    }
    elseif ($selected_portal) {
      $selected_portal = strtolower($selected_portal);
      $url .= $selected_portal . '/';
    }

    $query_params = [];
    $has_filters = FALSE;
    $request = $this->requestStack->getCurrentRequest();

    $authidValues = '';

    if ($request && $request->query->get('author')) {
      $author_param = $request->query->get('author');
      $author_ids = explode("+", $author_param);
      $tabauthids = [];
      foreach ($author_ids as $hal_id) {
        $hal_id = trim($hal_id);
        if (!empty($hal_id)) {
          $normalized_id = str_replace(' ', '', $hal_id);
          $tabauthids[$normalized_id] = $normalized_id;
        }
      }
    }
    else {
      $tabauthids = $this->getHalIds();
    }

    if (empty($tabauthids)) {
      $query_params['q'] = 'halId_s:__NONE__';
      $query_params['rows'] = '0';
      $url .= '?' . http_build_query($query_params);
      return $url;
    }

    $hal_id_values = array_values($tabauthids);
    $this->logger->info('Building query with @count HAL IDs (@ids)', [
      '@count' => count($hal_id_values),
      '@ids' => implode(', ', array_slice($hal_id_values, 0, 5)) . (count($hal_id_values) > 5 ? '...' : ''),
    ]);

    if (count($hal_id_values) > 1) {
      $authidValues = '(' . implode(' OR ', $hal_id_values) . ')';
    }
    else {
      $authidValues = reset($hal_id_values) ?: '';
    }

    $author_field = $this->getFilterFieldName('author');
    if ($author_field) {
      $fq_value = $author_field . ':' . $authidValues;
      $query_params['fq'][] = $fq_value;
      $has_filters = TRUE;
      $this->logger->info('Added author filter: @fq', ['@fq' => $fq_value]);
    }
    else {
      $this->logger->error('Author field name not found. This should not happen.');
    }

    if ($request && $request->query->get('year')) {
      $year_field = $this->getFilterFieldName('year');
      if ($year_field) {
        $yearsArr = explode("+", $request->query->get('year'));
        if (count($yearsArr) > 1) {
          $years = '(' . implode(' OR ', $yearsArr) . ')';
        }
        else {
          $years = implode('', $yearsArr);
        }
        $query_params['fq'][] = $year_field . ':' . $years;
        $has_filters = TRUE;
      }
    }

    if ($request && $request->query->get('term')) {
      $text_field = $this->getFilterFieldName('text');
      if ($text_field) {
        $query_params['fq'][] = $text_field . ':' . $request->query->get('term');
        $has_filters = TRUE;
      }
    }

    if ($is_rows) {
      $query_params['rows'] = '0';
    }
    else {
      $rows = $this->getDefaultRows();
      if ($rows <= 0) {
        $rows = 10000;
      }
      if ($rows > 10000) {
        $rows = 10000;
      }
      if ($rows > 0) {
        $query_params['rows'] = (string) $rows;
      }
    }

    if ($has_filters && !isset($query_params['q'])) {
      $query_params['q'] = '*:*';
    }

    $api_fields = $this->getApiFields();
    if ($api_fields) {
      $query_params['fl'] = $api_fields;
    }

    $sort = $this->getSort($config);
    if ($sort) {
      $query_params['sort'] = $sort;
    }

    $fq_params = $query_params['fq'] ?? [];
    unset($query_params['fq']);

    if (!empty($fq_params)) {
      $url .= '?fq=' . urlencode($fq_params[0]);
      for ($i = 1; $i < count($fq_params); $i++) {
        $url .= '&fq=' . urlencode($fq_params[$i]);
      }
    }

    if (isset($query_params['rows'])) {
      $url .= (strpos($url, '?') !== FALSE ? '&' : '?') . 'rows=' . $query_params['rows'];
    }

    if (isset($query_params['fl'])) {
      $url .= '&fl=' . $query_params['fl'];
    }

    if (isset($query_params['sort'])) {
      $url .= '&sort=' . $query_params['sort'];
    }

    if (isset($query_params['start'])) {
      $url .= '&start=' . $query_params['start'];
    }

    $this->logger->info('Generated HAL API URL: @url', ['@url' => $url]);

    if (empty($tabauthids)) {
      $this->logger->warning('No HAL IDs found for query. Generated URL: @url', ['@url' => $url]);
    }

    return $url;
  }

  /**
   * Get filters for the filter form.
   *
   * @param array $config
   *   Configuration array.
   *
   * @return array
   *   Filters values.
   */
  public function getFilters(array $config) {
    $filters = [];

    $filters['author'] = $this->getHalIds(TRUE);
    $year_field = $this->getFilterFieldName('year');
    $filters['year'] = $this->getApiFacets($year_field, TRUE);

    if (isset($config['multiple_authors'][1]) && $config['multiple_authors'][1] == 0) {
      $filters['author'] = ["" => $this->t("Author")] + $filters['author'];
    }
    if (isset($config['multiple_years'][1]) && $config['multiple_years'][1] == 0) {
      $filters['year'] = ["" => $this->t("Year")] + $filters['year'];
    }

    return $filters;
  }

  /**
   * Retrieve HAL IDs from HAL Author entities.
   *
   * @param bool $plusNames
   *   If TRUE, returns author names instead of raw HAL IDs.
   *
   * @return array
   *   HAL IDs (or names) keyed by entity_id or HAL ID.
   */
  public function getHalIds(bool $plusNames = FALSE): array {
    try {
      $hal_author_storage = $this->entityTypeManager->getStorage('hal_author');

      $query = $hal_author_storage->getQuery()
        ->accessCheck(FALSE)
        ->condition('status', 1)
        ->exists('hal_id');

      $author_ids = $query->execute();

      if (empty($author_ids)) {
        return [];
      }

      /** @var \Drupal\hal_publications\Entity\HalAuthor[] $authors */
      $authors = $hal_author_storage->loadMultiple($author_ids);
      $all_ids = [];

      $processed_count = 0;
      $skipped_count = 0;
      foreach ($authors as $author_id => $author) {
        $hal_id = $author->getHalId();

        if (empty($hal_id)) {
          $skipped_count++;
          continue;
        }

        $hal_value = str_replace(' ', '', $hal_id);
        $processed_count++;

        if ($plusNames) {
          $name = trim($author->label());
          $all_ids[$hal_value] = $name !== '' ? $name : $hal_value;
        }
        else {
          $all_ids[$author_id] = $hal_value;
        }
      }

      asort($all_ids);

      if (count($all_ids) > 0) {
        $this->logger->info('Loaded HAL IDs: @count IDs found (@ids)', [
          '@count' => count($all_ids),
          '@ids' => implode(', ', array_slice(array_values($all_ids), 0, 5)) . (count($all_ids) > 5 ? '...' : ''),
        ]);
      }
      else {
        $this->logger->warning('No HAL IDs loaded. Check if HAL Author entities exist and are published.');
      }

      return $all_ids;
    }
    catch (\Exception $e) {
      $this->logger->error('Error loading HAL IDs: @message', ['@message' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * Get facets from HAL API.
   *
   * @param string $facet
   *   Facet field.
   * @param bool $reverse
   *   TRUE: results sorted in descending order.
   *
   * @return array
   *   Facets.
   */
  public function getApiFacets(string $facet, bool $reverse = FALSE): array {
    $clean_facets = [];
    $url = $this->getFacetsUrl($facet);
    $result = $this->request($url, TRUE);

    $unclean_facets = $result['facet_counts']['facet_fields'][$facet] ?? [];

    if (!is_array($unclean_facets)) {
      return [];
    }

    foreach ($unclean_facets as $key => $value) {
      if ($key % 2 !== 0) {
        continue;
      }
      if (isset($unclean_facets[$key + 1]) && $unclean_facets[$key + 1] != 0) {
        $clean_facets[$value] = $value;
      }
    }

    if ($reverse) {
      arsort($clean_facets);
    }
    else {
      asort($clean_facets);
    }

    return $clean_facets;
  }

  /**
   * Creates facets URL.
   *
   * @param string $field
   *   Facet field.
   *
   * @return string
   *   URL string.
   */
  public function getFacetsUrl(string $field): string {
    $base_url = rtrim(self::HAL_API_BASE_URL, '/');

    // Facets URL doesn't use portal/collection - searches all HAL.
    $url = $base_url . '/search/';

    $query_params = [];
    $tabauthids = $this->getHalIds();
    $authidValues = '';

    if (count($tabauthids) > 0) {
      $authidValues = '(' . implode(' OR ', array_values($tabauthids)) . ')';
      $author_field = $this->getFilterFieldName('author');
      if ($author_field) {
        $query_params['fq'][] = $author_field . ':' . $authidValues;
        $query_params['q'] = '*:*';
      }
    }

    $query_params['rows'] = '0';
    $query_params['facet'] = 'true';
    $query_params['facet.field'] = $field;

    // Add facet options.
    $facet_sort = $this->getFacetSort();
    if ($facet_sort) {
      $query_params['facet.sort'] = $facet_sort;
    }

    $facet_limit = $this->getFacetLimit();
    if ($facet_limit > 0) {
      $query_params['facet.limit'] = (string) $facet_limit;
    }

    // Handle fq parameters separately (same as generateUrl).
    $fq_params = $query_params['fq'] ?? [];
    unset($query_params['fq']);

    $query_string = http_build_query($query_params, '', '&', PHP_QUERY_RFC3986);

    // Add fq parameters separately.
    if (!empty($fq_params)) {
      foreach ($fq_params as $fq_value) {
        $query_string .= '&fq=' . rawurlencode($fq_value);
      }
    }

    $url .= '?' . $query_string;
    return $url;
  }

  /**
   * Get module path.
   *
   * @return string
   *   Module path.
   */
  public function getModulePath(): string {
    return $this->moduleExtensionList->getPath('hal_publications');
  }

  /**
   * Get file system service.
   *
   * @return \Drupal\Core\File\FileSystemInterface
   *   File system.
   */
  public function getFileSystem(): FileSystemInterface {
    return $this->fileSystem;
  }

  /**
   * Gets portals from configuration.
   *
   * @return array
   *   Array of portals with 'id' and 'label' keys.
   */
  public function getPortals(): array {
    $config = $this->configFactory->get('hal_publications.settings');
    return $config->get('portals') ?: [];
  }

  /**
   * Gets portal options as key-value pairs (id => label).
   *
   * @return array
   *   Array of portal_id => portal_label.
   */
  public function getPortalOptions(): array {
    $portals = $this->getPortals();
    $options = [];
    foreach ($portals as $portal) {
      if (!empty($portal['id']) && !empty($portal['label'])) {
        $options[$portal['id']] = $portal['label'];
      }
    }
    return $options;
  }

  /**
   * Gets collections from configuration.
   *
   * @return array
   *   Array of collections with 'id' and 'label' keys.
   */
  public function getCollections(): array {
    $config = $this->configFactory->get('hal_publications.settings');
    return $config->get('collections') ?: [];
  }

  /**
   * Gets collection options as key-value pairs (id => label).
   *
   * @return array
   *   Array of collection_id => collection_label.
   */
  public function getCollectionOptions(): array {
    $collections = $this->getCollections();
    $options = [];
    foreach ($collections as $collection) {
      if (!empty($collection['id']) && !empty($collection['label'])) {
        $options[$collection['id']] = $collection['label'];
      }
    }
    return $options;
  }

  /**
   * Gets the HAL API base URL.
   */
  public function getHalApiBaseUrl(): string {
    return self::HAL_API_BASE_URL;
  }

}
