<?php

declare(strict_types=1);

namespace Drupal\halloween_lights\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Settings form for the Halloween Lights module.
 *
 * @package Drupal\halloween_lights\Form
 */
class HalloweenLightsSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'halloween_lights_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'halloween_lights.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('halloween_lights.settings');

    // Build associative options: 1 => 1, 2 => 2, ...
    $days_october  = array_combine(range(1, 31), range(1, 31));
    $days_november = array_combine(range(1, 30), range(1, 30));

    $form['period'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Working period'),
      '#attached' => [
        'library' => [
          'halloween_lights/settings-form',
        ],
      ],
    ];
    $form['period']['halloween_lights_period'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable working period'),
      '#default_value' => $config->get('halloween_lights_period'),
      '#description' => $this->t('If the working period is not specified, the module will run all the time.'),
    ];
    $form['period']['period_dates'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Select start date and end date'),
      '#states' => [
        'visible' => [
          '#edit-halloween-lights-period' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['period']['period_dates']['halloween_lights_start'] = [
      '#type' => 'select',
      '#title' => $this->t('Start day (October)'),
      '#options' => $days_october,
      '#default_value' => $config->get('halloween_lights_start'),
    ];
    $form['period']['period_dates']['halloween_lights_end'] = [
      '#type' => 'select',
      '#title' => $this->t('End day (November)'),
      '#options' => $days_november,
      '#default_value' => $config->get('halloween_lights_end'),
    ];
    $form['lights_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Halloween Lights Placement'),
    ];
    $form['lights_settings']['halloween_lights_topfixed'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Top-Fixed lights'),
      '#default_value' => $config->get('halloween_lights_topfixed'),
      '#description' => $this->t('Keep the lights fixed at the top when scrolling.'),
    ];
    $form['lights_settings']['halloween_lights_coretoolbar'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Drupal core toolbar is used (add lights after toolbar)'),
      '#default_value' => $config->get('halloween_lights_coretoolbar'),
      '#description' => $this->t('Position lights below the Drupal admin toolbar.'),
    ];
    $form['lights_settings']['halloween_lights_bootstrapfixed'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Bootstrap navbar-fixed is used (add lights after navbar)'),
      '#default_value' => $config->get('halloween_lights_bootstrapfixed'),
      '#description' => $this->t('Position lights below Bootstrap fixed navbar.'),
    ];
    $form['lights_settings']['halloween_lights_custommargin'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use custom top margin'),
      '#default_value' => $config->get('halloween_lights_custommargin'),
    ];
    $form['lights_settings']['halloween_lights_custommargintext'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Enter custom top margin (in pixels)'),
      '#default_value' => $config->get('halloween_lights_custommargintext'),
      '#maxlength' => 7,
      '#size' => 7,
      '#states' => [
        'visible' => [
          '#edit-halloween-lights-custommargin' => ['checked' => TRUE],
        ],
      ],
    ];


    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $this->config('halloween_lights.settings')
      ->set('halloween_lights_period', $values['halloween_lights_period'])
      ->set('halloween_lights_start', $values['halloween_lights_start'])
      ->set('halloween_lights_end', $values['halloween_lights_end'])
      ->set('halloween_lights_topfixed', $values['halloween_lights_topfixed'])
      ->set('halloween_lights_coretoolbar', $values['halloween_lights_coretoolbar'])
      ->set('halloween_lights_bootstrapfixed', $values['halloween_lights_bootstrapfixed'])
      ->set('halloween_lights_custommargin', $values['halloween_lights_custommargin'])
      ->set('halloween_lights_custommargintext', $values['halloween_lights_custommargintext'])
      ->save();

    $this->messenger()->addMessage($this->t('The configuration options have been saved.'));
  }

  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $start = (int) $form_state->getValue('halloween_lights_start');
    $end   = (int) $form_state->getValue('halloween_lights_end');

    if ($start < 1 || $start > 31) {
      $form_state->setErrorByName('halloween_lights_start', $this->t('Start day must be between 1 and 31.'));
    }
    if ($end < 1 || $end > 30) {
      $form_state->setErrorByName('halloween_lights_end', $this->t('End day must be between 1 and 30.'));
    }
  }


}
