<?php

declare(strict_types=1);

/**
 * @file
 * Contains \Drupal\hello_login\HelloClient.
 *
 * Provides functionality to manage user login, creation, updates, and logout
 * within the HelloCoop module. It handles user data processing, profile picture
 * management, and invokes relevant events for custom extensions.
 */

namespace Drupal\hello_login;

use Drupal\file\FileInterface;
use Drupal\Core\File\FileExists;
use Drupal\file\FileRepositoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\user\Entity\User;
use Drupal\externalauth\ExternalAuthInterface;
use Drupal\user\UserInterface;
use GuzzleHttp\ClientInterface;

/**
 * Provides functionality for managing HelloCoop user login and logout.
 */
class HelloClient {

  /**
   * The User entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $userStorage;

  /**
   * The User entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $fileStorage;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  protected FileRepositoryInterface $fileRepository;

  /**
   * Logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The external auth.
   *
   * @var \Drupal\externalauth\ExternalAuthInterface
   */
  protected ExternalAuthInterface $externalAuth;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The file system service.
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Constructs a HelloClient object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\file\FileRepositoryInterface $fileRepository
   *   The file repository service for saving user profile pictures.
   * @param \Drupal\externalauth\ExternalAuthInterface $externalAuth
   *   The Interface for the ExternalAuth module.
   * @param GuzzleHttp\ClientInterface $httpClient
   *   HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   Logger.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system interface to access the underlying filesystem.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    ModuleHandlerInterface $moduleHandler,
    FileRepositoryInterface $fileRepository,
    ExternalAuthInterface $externalAuth,
    ClientInterface $httpClient,
    LoggerChannelFactoryInterface $loggerFactory,
    FileSystemInterface $fileSystem,
  ) {
    $this->userStorage = $entityTypeManager->getStorage('user');
    $this->fileStorage = $entityTypeManager->getStorage('file');
    $this->moduleHandler = $moduleHandler;
    $this->fileRepository = $fileRepository;
    $this->externalAuth = $externalAuth;
    $this->httpClient = $httpClient;
    $this->logger = $loggerFactory->get('hello_login');
    $this->fileSystem = $fileSystem;
  }

  /**
   * Logs in or updates a user based on the provided payload.
   *
   * @param array $payload
   *   An associative array containing user information.
   */
  public function loginUpdate(array $payload): void {
    $user = $this->loadUser($payload);

    if (!$user) {
      $user = $this->createUser($payload);
    }

    $this->updateUserFields($user, $payload);
    $user->save();
    $this->moduleHandler->invokeAll('hello_login_user_login', ['user' => $user]);

    $this->externalAuth->userLoginFinalize($user, $payload['sub'], 'hello_login');
  }

  /**
   * Logs out the current user.
   */
  public function logOut(): void {
    $this->moduleHandler->invokeAll('hello_login_user_logout', []);
    user_logout();
  }

  /**
   * Loads an existing user or creates a new one.
   *
   * @param array $payload
   *   The user data payload.
   *
   * @return \Drupal\user\Entity\User
   *   The loaded or created user entity.
   */
  private function loadUser(array $payload): bool | UserInterface {
    /** @var \Drupal\user\UserInterface|bool $account */
    return $this->externalAuth->load($payload['sub'], 'hello_login');
  }

  /**
   * Loads an existing user or creates a new one.
   *
   * @param array $payload
   *   The user data payload.
   *
   * @return \Drupal\user\Entity\User
   *   The loaded or created user entity.
   */
  private function createUser(array $payload): User {
    $userData = [
      'name' => $payload['name'],
      'mail' => $payload['email'],
      'init' => $payload['email'],
    // Active user.
      'status' => 1,
    ];
    return $this->externalAuth->register($payload['sub'], 'hello_login', $userData);
  }

  /**
   * Updates user fields based on the provided payload.
   *
   * @param \Drupal\user\Entity\User $user
   *   The user entity to update.
   * @param array $payload
   *   The user data payload.
   */
  private function updateUserFields(User $user, array $payload): void {

    if (isset($payload['name']) && !empty($payload['name'])) {
      $user->set('name', $payload['name']);
    }

    $accountByMail = $this->userStorage->loadByProperties(['mail' => $payload['email']]);
    $accountByMail = $accountByMail ? reset($accountByMail) : NULL;

    if (empty($accountByMail) || ($accountByMail->id() == $user->id())) {
      $user->set('mail', $payload['email']);
    }

    if (isset($payload['picture'])) {
      $this->updateUserPicture($user, $payload['picture']);
    }

    // phpcs:disable
    // Additional fields can be added here with proper validations.
    // Example:
    // if (isset($payload['phone_number']) &&
    //     $this->isValidPhoneNumber($payload['phone_number'])) {
    //   $user->set('field_phone_number', $payload['phone_number']);
    // }
    // phpcs:enable
  }

  /**
   * Validates phone number format.
   *
   * @param string $phoneNumber
   *   The phone number to validate.
   *
   * @return bool
   *   TRUE if the phone number is valid, FALSE otherwise.
   */
  private function isValidPhoneNumber(string $phoneNumber): bool {
    // Basic validation example (can be replaced with more robust logic).
    return preg_match('/^\+?[0-9]{10,15}$/', $phoneNumber) === 1;
  }

  /**
   * Updates the user's profile picture.
   *
   * @param \Drupal\user\Entity\User $user
   *   The user entity to update.
   * @param string $pictureUrl
   *   The URL of the profile picture.
   *
   * @throws \LogicException
   *   If the profile picture cannot be saved.
   */
  private function updateUserPicture(User $user, string $pictureUrl): void {
    $externalImage = $this->saveUserImageAsFile($pictureUrl);
    if ($externalImage && $externalImage->getFileUri()) {
      $file = $this->fileStorage->create([
        'uri' => $externalImage->getFileUri(),
      ]);

      $file->save();

      if ($file->id()) {
        $user->set('user_picture', $file->id());
      }
      else {
        throw new \LogicException('Failed to save file entity or retrieve its ID.');
      }
    }
    else {
      throw new \LogicException('Failed to save external image as file.');
    }
  }

  /**
   * Saves an external image as a file.
   *
   * @param string $url
   *   The external image URL.
   *
   * @return \Drupal\file\FileInterface|null
   *   The saved file entity or NULL on failure.
   */
  private function saveUserImageAsFile(string $url): ?FileInterface {
    try {
      $data = (string) $this->httpClient->get($url)->getBody();
      $directory = 'public://user_pictures';

      // Ensure the directory exists and is writable.
      $prepared = $this->fileSystem->prepareDirectory(
        $directory,
        FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
      );

      if (!$prepared) {
        $this->logger->error('Failed to prepare directory @dir for user pictures.', ['@dir' => $directory]);
        // Return early if prepareDirectory failed.
        return NULL;
      }

      return $this->fileRepository->writeData(
        $data,
        $directory . '/' . uniqid('profile_', TRUE) . '.jpg',
        FileExists::Replace
      );
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to save external image: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

}
