<?php

declare(strict_types=1);

/**
 * @file
 * Contains \Drupal\hello_login\HelloConfigFactory.
 *
 * Provides a factory service class for creating configured HelloConfig objects.
 * It handles dependency injection and implements login and logout callback
 * functionalities used by the HelloCoop module.
 */

namespace Drupal\hello_login;

use HelloCoop\Config\HelloConfig;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Config;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Factory class for creating HelloConfig objects.
 *
 * This class creates instances of the HelloConfig class, configures them with
 * API routes, app ID, and secret values, and injects dependencies for callback
 * functions used for login and logout functionalities.
 */
class HelloConfigFactory {

  /**
   * The HelloClient service.
   *
   * @var \Drupal\hello_login\HelloClient
   */
  protected HelloClient $helloClient;

  /**
   * The HelloCoop configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected Config $config;

  /**
   * Request Stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a HelloConfigFactory object.
   *
   * @param \Drupal\hello_login\HelloClient $helloClient
   *   The HelloClient service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   */
  public function __construct(
    HelloClient $helloClient,
    ConfigFactoryInterface $configFactory,
    RequestStack $requestStack,
  ) {
    $this->helloClient = $helloClient;
    $this->config = $configFactory->get('hello_login.settings');
    $this->requestStack = $requestStack;
  }

  /**
   * Login synchronous callback function.
   *
   * @param array $helloWalletResponse
   *   The response data from the Hello Wallet service.
   *
   * @return array
   *   The modified or original response data.
   */
  public function loginCallback(array $helloWalletResponse = []): array {
    if (empty($helloWalletResponse['payload']) || empty($helloWalletResponse['payload']['email'])) {
      return $helloWalletResponse;
    }

    $this->helloClient->loginUpdate($helloWalletResponse['payload']);
    return $helloWalletResponse;
  }

  /**
   * Logout synchronous callback function.
   */
  public function logoutCallback(): void {
    $this->helloClient->logOut();
  }

  /**
   * Creates a configured HelloConfig instance.
   *
   * @return \HelloCoop\Config\HelloConfig
   *   A configured HelloConfig object.
   */
  public function createConfig(): HelloConfig {
    $request = $this->requestStack->getCurrentRequest();
    return new HelloConfig(
      $this->config->get('api_route'),
      $this->config->get('api_route') . '?op=auth',
      $this->config->get('api_route') . '?op=login',
      $this->config->get('api_route') . '?op=logout',
      FALSE,
      $this->config->get('app_id'),
      $request->getSchemeAndHttpHost() . $this->config->get('api_route'),
      $request->getHost(),
      $this->config->get('secret'),
      [$this, 'loginCallback'],
      [$this, 'logoutCallback']
    );
  }

}
