<?php

declare(strict_types=1);

namespace Drupal\hello_login\HelloResponse;

use HelloCoop\HelloResponse\HelloResponseInterface;
use Drupal\Core\Render\HtmlResponse;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Cookie;

/**
 * Handles setting headers, cookies, and returning proper responses.
 */
final class HelloResponse implements HelloResponseInterface {
  /**
   * Headers from the response.
   *
   * @var array<string,string>
   */
  protected array $headers = [];

  /**
   * Cookies from the response.
   *
   * @var \Symfony\Component\HttpFoundation\Cookie[]
   */
  protected array $cookies = [];

  /**
   * HTTP Status code from the response.
   *
   * @var int
   */
  protected int $statusCode = 200;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function setHeader(string $name, $value): void {
    $this->headers[$name] = is_array($value) ? implode(', ', $value) : (string) $value;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function deleteCookie(string $name, string $path = '/', string $domain = ''): void {
    $this->cookies[] = new Cookie($name, '', time() - 3600, $path, $domain, FALSE, TRUE, FALSE, 'Lax');
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function setCookie(
    string $name,
    string $value,
    int $expire = 0,
    string $path = '/',
    string $domain = '',
    bool $secure = FALSE,
    bool $httponly = TRUE,
  ): void {
    $this->cookies[] = new Cookie($name, $value, $expire ?: 0, $path, $domain, $secure, $httponly, FALSE, 'Lax');
  }

  /**
   * Apply headers, cookies, and the current status code to the response.
   */
  protected function finalize(Response $response): Response {
    // Status code first (so callers can override later if they really want).
    $response->setStatusCode($this->statusCode);

    foreach ($this->headers as $name => $value) {
      $response->headers->set($name, $value);
    }
    foreach ($this->cookies as $cookie) {
      $response->headers->setCookie($cookie);
    }
    return $response;
  }

  /**
   * Set the HTTP status code for subsequent responses.
   */
  #[\Override]
  public function setStatusCode(int $code): void {
    $this->statusCode = $code;
  }

  /**
   * Send an empty response immediately.
   *
   * Not typical in Drupal; prefer returning a Response.
   */
  #[\Override]
  public function send(): void {
    // Create an empty response with the currently selected status,
    // apply headers/cookies, then send.
    $response = $this->finalize(new Response(''));
    $response->send();
  }

  /**
   * Return a trusted redirect (allows external URLs).
   */
  #[\Override]
  public function redirect(string $url): Response {
    // Default 302; if you've set a custom status
    // (e.g., 301/307), finalize() applies it.
    return $this->finalize(new TrustedRedirectResponse($url));
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function render(string $content): Response {
    return $this->finalize(new HtmlResponse($content));
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function json(array $data): Response {
    return $this->finalize(new JsonResponse($data));
  }

}
