<?php

declare(strict_types=1);

namespace Drupal\hello_login\HelloRequest;

use HelloCoop\HelloRequest\HelloRequestInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * For fetching cookies and HTTP request params.
 */
final class HelloRequest implements HelloRequestInterface {
  /**
   * The current request (may be NULL in CLI).
   */
  private ?Request $currentRequest;

  public function __construct(RequestStack $requestStack) {
    $this->currentRequest = $requestStack->getCurrentRequest();
  }

  /**
   * Check if a parameter exists in either GET or POST data.
   */
  #[\Override]
  public function has(string $key): bool {
    $r = $this->currentRequest;
    if (!$r) {
      return FALSE;
    }
    // GET: $r->query, POST: $r->request.
    return $r->query->has($key) || $r->request->has($key);
  }

  /**
   * Prefer POST over GET (typical semantics), then fall back to $default.
   */
  #[\Override]
  public function fetch(string $key, $default = NULL): ?string {
    $r = $this->currentRequest;
    if (!$r) {
      return $default;
    }

    if ($r->request->has($key)) {
      $val = $r->request->get($key);
      return $val !== NULL ? (string) $val : $default;
    }

    if ($r->query->has($key)) {
      $val = $r->query->get($key);
      return $val !== NULL ? (string) $val : $default;
    }

    return $default;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function fetchMultiple(array $keys): array {
    $r = $this->currentRequest;
    $values = [];

    foreach ($keys as $key) {
      if (!$r) {
        $values[$key] = NULL;
        continue;
      }

      if ($r->request->has($key)) {
        $values[$key] = $r->request->get($key);
      }
      elseif ($r->query->has($key)) {
        $values[$key] = $r->query->get($key);
      }
      else {
        $values[$key] = NULL;
      }
    }

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function fetchHeader(string $key, $default = NULL): ?string {
    $r = $this->currentRequest;
    if (!$r) {
      return $default;
    }
    $val = $r->headers->get($key, $default);
    return $val !== NULL ? (string) $val : $default;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getCookie(string $name): ?string {
    $r = $this->currentRequest;
    if (!$r) {
      return NULL;
    }
    $val = $r->cookies->get($name, NULL);
    return $val !== NULL ? (string) $val : NULL;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getRequestUri(): string {
    return $this->currentRequest?->getRequestUri() ?? '';
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getMethod(): string {
    return $this->currentRequest?->getMethod() ?? '';
  }

}
