<?php

declare(strict_types=1);

namespace Drupal\helpdesk_integration\Plugin\TopBarItem;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\helpdesk_integration\Service;
use Drupal\navigation\Attribute\TopBarItem;
use Drupal\navigation\TopBarItemBase;
use Drupal\navigation\TopBarRegion;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the test top bar item.
 *
 * @internal
 */
#[TopBarItem(
  id: 'helpdesk_integration_tools',
  region: TopBarRegion::Tools,
  label: new TranslatableMarkup('Helpdesk Integration Tools'),
)]
final class HelpdeskTools extends TopBarItemBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The helpdesk service.
   *
   * @var \Drupal\helpdesk_integration\Service
   */
  protected Service $helpdeskService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): HelpdeskTools {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->helpdeskService = $container->get('helpdesk_integration.service');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $links = $this->helpdeskService->getToolbarLinks();
    if (empty($links)) {
      return [];
    }
    $featuredLink = array_shift($links);
    $featuredLink['#attributes']['class'][] = 'top-bar-featured-hide-label';
    $featuredLinks['featured'] = [
      'page_action' => [
        '#theme' => 'top_bar_page_action',
        '#link' => $featuredLink,
      ],
      'icon' => [
        'icon_id' => 'database',
      ],
    ];
    foreach ($links as $key => $link) {
      $links[$key] = [
        '#theme' => 'top_bar_page_action',
        '#link' => $link,
      ];
    }

    return [
      '#cache' => [
        'contexts' => ['user.permissions'],
      ],
      '#theme' => 'top_bar_page_actions',
      '#page_actions' => $links,
      '#featured_page_actions' => $featuredLinks,
      '#attached' => [
        'library' => [
          'core/drupal.dialog.ajax',
          'helpdesk_integration/toolbar',
        ],
      ],
    ];
  }

}
