<?php

namespace Drupal\hierarchical_select_ajax\Plugin\Field\FieldWidget;

use Drupal\Core\Config\Entity\ThirdPartySettingsInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\EntityReferenceAutocompleteWidget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\hierarchical_select_ajax\HierarchicalSelectHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides "hierarchical_select_ajax" field widget.
 *
 * @FieldWidget(
 *   id = "hierarchical_select_ajax",
 *   label = @Translation("Hierarchical select ajax"),
 *   field_types = {
 *     "entity_reference",
 *   },
 * )
 */
class HierarchicalSelectAjaxWidget extends EntityReferenceAutocompleteWidget {

  use HierarchicalSelectHelper {
    defaultSettings as helperDefaultSettings;
    formElement as helperFormElement;
    settingsForm as helperSettingsForm;
  }

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, ModuleHandlerInterface $module_handler) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function handlesMultipleValues() {
    // Pretend to support multiple values to get a single element
    // if field_config_cardinality == 1.
    if ($this->fieldDefinition instanceof ThirdPartySettingsInterface
      && $this->fieldDefinition->getThirdPartySetting('field_config_cardinality', 'cardinality_config') == 1) {
      return TRUE;
    }

    return parent::handlesMultipleValues();
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    // Extract single value from fake single element.
    if ($this->fieldDefinition instanceof ThirdPartySettingsInterface
      && $this->fieldDefinition->getThirdPartySetting('field_config_cardinality', 'cardinality_config') == 1) {
      return [$values['target_id']];
    }

    return parent::massageFormValues($values, $form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition): bool {
    if ($field_definition->getFieldStorageDefinition()->getSetting('target_type') === 'taxonomy_term') {
      /* @see \Drupal\Core\Entity\Plugin\EntityReferenceSelection\DefaultSelection */
      if (!empty($field_definition->getSettings()['handler_settings']['target_bundles'] ?? [])) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return static::helperDefaultSettings() + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    return $this->helperSettingsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * @see hierarchical_select_ajax_formatter_element_alter()
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $hsa_element = parent::formElement($items, $delta, $element, $form, $form_state);

    // Context info for the element to be altered.
    $hsa_info = [
      'field_name' => $this->fieldDefinition->getName(),
      'entity_type' => $this->fieldDefinition->getTargetEntityTypeId(),
      'bundle' => $this->fieldDefinition->getTargetBundle(),
      'entity' => $items->getEntity(),
    ];

    // Allow other modules to alter the element.
    $this->moduleHandler->alter('hierarchical_select_ajax_element', $hsa_element, $hsa_info);

    // Allow other modules to alter the hierarchical select element settings.
    $settings = $this->getSetting('hierarchical_select_ajax');
    $this->moduleHandler->alter('hierarchical_select_ajax_settings', $settings, $hsa_info);

    // Initialize the hierarchical select AJAX element.
    $element = $this->helperFormElement($hsa_element, $settings);

    return $element;
  }

}
