<?php

declare(strict_types=1);

namespace Drupal\Tests\highlightjs_input_filter\Unit;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\highlightjs_input_filter\Plugin\Filter\HighlightJs;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

#[CoversClass(HighlightJs::class)]
#[Group('highlightjs_input_filter')]
class HighlightJsAliasResolutionTest extends UnitTestCase {

  protected function buildFilter(): HighlightJs {
    $config_factory = $this->createMock(ConfigFactoryInterface::class);
    $plugin_definition = ['provider' => 'highlightjs_input_filter'];
    return new HighlightJs([], 'filter_highlightjs', $plugin_definition, $config_factory);
  }

  /**
   * Invokes the protected resolveAlias() method via reflection.
   */
  protected function resolve(HighlightJs $filter, string $lang): string {
    $method = new \ReflectionMethod($filter, 'resolveAlias');
    $method->setAccessible(true);
    return $method->invoke($filter, $lang);
  }

  public function testAliasResolutionForJavascriptVariants(): void {
    $filter = $this->buildFilter();
    $this->assertSame('javascript', $this->resolve($filter, 'js'));
    $this->assertSame('javascript', $this->resolve($filter, 'jsx'));
    // Already canonical should stay the same.
    $this->assertSame('javascript', $this->resolve($filter, 'javascript'));
  }

  public function testAliasResolutionForXmlVariants(): void {
    $filter = $this->buildFilter();
    $this->assertSame('xml', $this->resolve($filter, 'html'));
    $this->assertSame('xml', $this->resolve($filter, 'xhtml'));
    $this->assertSame('xml', $this->resolve($filter, 'rss'));
    $this->assertSame('xml', $this->resolve($filter, 'atom'));
    $this->assertSame('xml', $this->resolve($filter, 'xjb'));
    $this->assertSame('xml', $this->resolve($filter, 'xsd'));
    $this->assertSame('xml', $this->resolve($filter, 'xsl'));
    $this->assertSame('xml', $this->resolve($filter, 'plist'));
    $this->assertSame('xml', $this->resolve($filter, 'svg'));
  }

  public function testAliasResolutionForCSharp(): void {
    $filter = $this->buildFilter();
    $this->assertSame('csharp', $this->resolve($filter, 'cs'));
    $this->assertSame('csharp', $this->resolve($filter, 'csharp'));
  }

  public function testUnknownLanguageIsReturnedAsIs(): void {
    $filter = $this->buildFilter();
    $this->assertSame('python', $this->resolve($filter, 'python'));
  }

}
