<?php

declare(strict_types=1);

namespace Drupal\Tests\highlightjs_input_filter\Unit;

// Load the procedural function under test.
require_once __DIR__ . '/../../../highlightjs_input_filter.module';

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use PHPUnit\Framework\Attributes\CoversFunction;
use PHPUnit\Framework\Attributes\Group;

#[CoversFunction('highlightjs_input_filter_library_info_build')]
#[Group('highlightjs_input_filter')]
class LibraryInfoBuildTest extends UnitTestCase {

  /**
   * Sets up a container with a mocked config for the given theme.
   */
  protected function setConfigTheme(string $theme): void {
    $container = new ContainerBuilder();

    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')
      ->with('theme')
      ->willReturn($theme);

    $config_factory = $this->createMock(ConfigFactoryInterface::class);
    $config_factory->method('get')
      ->with('highlightjs_input_filter.settings')
      ->willReturn($config);

    $container->set('config.factory', $config_factory);
    \Drupal::setContainer($container);
  }

  public function testDefaultThemeFallbackBuildsExternalCssUrl(): void {
    // An empty theme should fall back to the default.
    $this->setConfigTheme('');
    $libraries = highlightjs_input_filter_library_info_build();

    $this->assertArrayHasKey('highlightjs_input_filter.styles', $libraries);
    $css = $libraries['highlightjs_input_filter.styles']['css']['theme'] ?? [];
    $this->assertNotEmpty($css, 'Theme css array present');

    $url = array_key_first($css);
    $this->assertStringContainsString('https://unpkg.com/@highlightjs/cdn-assets@11.9.0/styles/', $url);
    // Default is atom-one-dark.
    $this->assertStringEndsWith('atom-one-dark.min.css', $url);

    $this->assertSame([
      'type' => 'external',
      'minified' => TRUE,
    ], $css[$url]);
  }

  public function testBase16HyphenatedThemeIsNormalizedToSubdir(): void {
    $this->setConfigTheme('base16-3024');
    $libraries = highlightjs_input_filter_library_info_build();
    $css = $libraries['highlightjs_input_filter.styles']['css']['theme'];
    $url = array_key_first($css);

    // base16-3024 -> base16/3024.min.css
    $this->assertStringEndsWith('base16/3024.min.css', $url);
  }

  public function testCanonicalPathThemeIsUsedAsIs(): void {
    $this->setConfigTheme('base16/solarized-dark');
    $libraries = highlightjs_input_filter_library_info_build();
    $css = $libraries['highlightjs_input_filter.styles']['css']['theme'];
    $url = array_key_first($css);

    $this->assertStringEndsWith('base16/solarized-dark.min.css', $url);
  }

  public function testFlatThemeIsUsedAsIs(): void {
    $this->setConfigTheme('atom-one-light');
    $libraries = highlightjs_input_filter_library_info_build();
    $css = $libraries['highlightjs_input_filter.styles']['css']['theme'];
    $url = array_key_first($css);

    $this->assertStringEndsWith('atom-one-light.min.css', $url);
  }

}
