<?php

namespace Drupal\hijri_format\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\hijri_format\HijriFormatManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'hijri_dual_date_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "hijri_dual_date_formatter",
 *   label = @Translation("Hijri Dual Date Formatter"),
 *   field_types = {
 *     "timestamp",
 *     "datetime",
 *     "daterange",
 *     "created",
 *     "changed"
 *   }
 * )
 */
class HijriDualDateFormatter extends FormatterBase {

  /**
   * Hijri manager.
   *
   * @var \Drupal\hijri_format\HijriFormatManager
   */
  protected $hijriFormatManager;

  /**
   * {@inheritdoc}
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, HijriFormatManager $hijri_format_manager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->hijriFormatManager = $hijri_format_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('hijri_format.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'format' => 'd-F-Y',
      'separator' => 'To',
      'is_indian' => 0,
      'from_to' => 'both',
      'label_start' => 'From',
      'label_end' => 'To',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function getFromToOptions() {
    return [
      'start' => $this->t('Start date only'),
      'end' => $this->t('End date only'),
      'both' => $this->t('Both start and end dates'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['format'] = [
      '#type' => 'select',
      '#title' => $this->t('Date Format'),
      '#options' => [
        'd-F-Y' => 'd-F-Y',
        'F-d-Y' => 'F-d-Y',
        'Y-F-d' => 'Y-F-d',
      ],
      '#default_value' => $this->getSetting('format'),
    ];

    $elements['from_to'] = [
      '#type' => 'select',
      '#title' => $this->t('Display'),
      '#options' => $this->getFromToOptions(),
      '#default_value' => $this->getSetting('from_to'),
    ];

    $elements['separator'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Separator'),
      '#default_value' => $this->getSetting('separator'),
    ];

    $elements['is_indian'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Indian Hijri Calendar'),
      '#default_value' => $this->getSetting('is_indian'),
    ];

    $elements['label_start'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Start Date Label'),
      '#default_value' => $this->getSetting('label_start'),
    ];

    $elements['label_end'] = [
      '#type' => 'textfield',
      '#title' => $this->t('End Date Label'),
      '#default_value' => $this->getSetting('label_end'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->t('Format: @format', ['@format' => $this->getSetting('format')]);
    $summary[] = $this->t('Separator: @separator', ['@separator' => $this->getSetting('separator')]);
    $summary[] = $this->t('Indian Calendar: @is_indian', ['@is_indian' => $this->getSetting('is_indian') ? 'Yes' : 'No']);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      try {
        $format = $this->getSetting('format');
        $is_indian = (bool) $this->getSetting('is_indian');
        $separator = $this->getSetting('separator');
        $from_to = $this->getSetting('from_to');
        $label_start = trim($this->getSetting('label_start'));
        $label_end = trim($this->getSetting('label_end'));

        $date_value = $item->getValue();
        if (empty($date_value)) {
          continue;
        }

        // Extract start and end values. // @sig
        $start_value = $date_value['value'] ?? NULL;
        $end_value = $date_value['end_value'] ?? NULL;

        if ($start_value) {
          $start_date = is_numeric($start_value)
            ? \DateTime::createFromFormat('U', $start_value, new \DateTimeZone('UTC'))
            : new \DateTime($start_value, new \DateTimeZone('UTC'));
        }
        if ($end_value) {
          $end_date = is_numeric($end_value)
            ? \DateTime::createFromFormat('U', $end_value, new \DateTimeZone('UTC'))
            : new \DateTime($end_value, new \DateTimeZone('UTC'));
        }

        if (empty($start_date)) {
          continue;
        }

        // Convert to Hijri. // @sig
        $start_hijri = $this->hijriFormatManager->convertToHijri($start_date->getTimestamp(), $format, $is_indian);
        $start_gregorian = $this->hijriFormatManager->dateFormatter->format($start_date->getTimestamp(), 'custom', $format);

        $end_hijri = '';
        $end_gregorian = '';

        if (!empty($end_date)) {
          $end_hijri = $this->hijriFormatManager->convertToHijri($end_date->getTimestamp(), $format, $is_indian);
          $end_gregorian = $this->hijriFormatManager->dateFormatter->format($end_date->getTimestamp(), 'custom', $format);
        }

        // Pass values to Twig template. // @sig
        $elements[$delta] = [
        // Custom template. // @sig
          '#theme' => 'hijri_dual_date_formatter',
          '#start_hijri' => $start_hijri,
          '#start_gregorian' => $start_gregorian,
          '#end_hijri' => $end_hijri,
          '#end_gregorian' => $end_gregorian,
          '#separator' => $separator,
          '#from_to' => $from_to,
          '#label_start' => $label_start,
          '#label_end' => $label_end,
          '#attributes' => ['class' => ['hijri-dual-date-formatter']],
        ];

      }
      catch (\Exception $e) {
        \Drupal::logger('hijri_dual_date_formatter')->error(
          'Error formatting date: @error (Value: @value)',
          [
            '@error' => $e->getMessage(),
            '@value' => print_r($date_value['value'] ?? 'N/A', TRUE),
          ]
        );
      }
    }

    return $elements;
  }

}
