<?php

namespace Drupal\hivo_connector\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\file\Entity\File;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileRepository;
use Drupal\media\Entity\Media;
use Drupal\media\MediaStorage;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\hivo_connector\Constants\HivoConstants;

class HivoUploadApiController extends ControllerBase {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepository
   */
  protected $fileRepository;

  /**
   * The media storage service.
   *
   * @var \Drupal\media\MediaStorage
   */
  protected $mediaStorage;

  /**
   * Constructs a new HivoUploadApiController.
   *
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\file\FileRepository $file_repository
   *   The file repository service.
   * @param \Drupal\media\MediaStorage $media_storage
   *   The media storage service.
   */
  public function __construct(FileSystemInterface $file_system, FileRepository $file_repository, MediaStorage $media_storage) {
    $this->fileSystem = $file_system;
    $this->fileRepository = $file_repository;
    $this->mediaStorage = $media_storage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('file_system'),
      $container->get('file.repository'),
      $container->get('entity_type.manager')->getStorage('media')
    );
  }

  /**
   * Detect media type based on file extension and MIME type.
   *
   * @param string $filename
   *   The filename.
   * @param string $mime_type
   *   The MIME type.
   *
   * @return array
   *   Array containing media type and bundle.
   */
  protected function detectMediaType($filename, $mime_type) {
    $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    
    // Image types
    $image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'bmp', 'tiff'];
    if (in_array($extension, $image_extensions) || strpos($mime_type, 'image/') === 0) {
      return [
        'type' => 'image',
        'bundle' => 'image',
        'field' => 'field_media_image'
      ];
    }
    
    // Video types
    $video_extensions = ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm', 'mkv', 'm4v'];
    if (in_array($extension, $video_extensions) || strpos($mime_type, 'video/') === 0) {
      return [
        'type' => 'video',
        'bundle' => 'video',
        'field' => 'field_media_video_file'
      ];
    }
    
    // Audio types
    $audio_extensions = ['mp3', 'wav', 'ogg', 'aac', 'flac', 'wma'];
    if (in_array($extension, $audio_extensions) || strpos($mime_type, 'audio/') === 0) {
      return [
        'type' => 'audio',
        'bundle' => 'audio',
        'field' => 'field_media_audio_file'
      ];
    }
    
    // Document types
    $document_extensions = ['pdf', 'doc', 'docx', 'txt', 'rtf', 'odt', 'xls', 'xlsx', 'ppt', 'pptx'];
    if (in_array($extension, $document_extensions) || strpos($mime_type, 'application/') === 0) {
      return [
        'type' => 'document',
        'bundle' => 'document',
        'field' => 'field_media_document'
      ];
    }
    
    // Default to file
    return [
      'type' => 'document',
      'bundle' => 'document',
      'field' => 'field_media_document'
    ];
  }

  public function upload(Request $request) {
    // Handle CORS preflight requests
    if ($request->getMethod() === 'OPTIONS') {
      $response = new JsonResponse();
      $response->headers->set('Access-Control-Allow-Origin', '*');
      $response->headers->set('Access-Control-Allow-Methods', 'POST, OPTIONS');
      $response->headers->set('Access-Control-Allow-Headers', 'Content-Type');
      return $response;
    }

    // Constants for validation
    $id_regex = '/^[23456789ABCDEFGHJKLMNPQRSTWXYZabcdefghijkmnopqrstuvwxyz]{17}$/';
    $token_regex = '/^[A-Za-z0-9+\/]{42}[AEIMQUYcgkosw048]=$/';

    // Get parameters from request
    $uploadId = $request->request->get('uploadId');
    $version = $request->request->get('version');
    $userId = $request->request->get('userId');
    $token = $request->request->get('token');

    // Validate required parameters with regex patterns
    if (!$uploadId) {
      return new JsonResponse([
        'error' => 'Bad Request: Missing uploadId'
      ], 400);
    }
    if (preg_match($id_regex, $uploadId) === false) {
      return new JsonResponse([
        'error' => 'Bad Request: Invalid uploadId format'
      ], 400);
    }
    if (!$version) {
      return new JsonResponse([
        'error' => 'Bad Request: Missing version'
      ], 400);
    }
    if (!$userId) {
      return new JsonResponse([
        'error' => 'Bad Request: Missing userId'
      ], 400);
    }
    if (preg_match($id_regex, $userId) === false) {
      return new JsonResponse([
        'error' => 'Bad Request: Invalid userId format'
      ], 400);
    }
    if (!$token) {
      return new JsonResponse([
        'error' => 'Bad Request: Missing token'
      ], 400);
    }
    if (preg_match($token_regex, $token) === false) {
      return new JsonResponse([
        'error' => 'Bad Request: Invalid token format'
      ], 400);
    }

    try {
      // Step 1: Download file from HIVO
      $hivoUpload = $this->downloadFromHivo($uploadId, $version, $userId, $token);
      
      if (!$hivoUpload || !isset($hivoUpload['url'])) {
        return new JsonResponse(['error' => 'Failed to get file URL from HIVO'], 500);
      }

      // Step 2: Download file content from URL
      $fileContent = $this->downloadFileContent($hivoUpload['url']);
      
      if (!$fileContent) {
        return new JsonResponse(['error' => 'Failed to download file content'], 500);
      }

      // Step 3: Save to Drupal
      $result = $this->saveToDrupal($fileContent, $hivoUpload['name'] ?? 'file', $hivoUpload);

      $response = new JsonResponse($result);
      $response->headers->set('Access-Control-Allow-Origin', '*');
      
      return $response;

    } catch (\Exception $e) {
      return new JsonResponse(['error' => 'Save failed: ' . $e->getMessage()], 500);
    }
  }

  /**
   * Download file information from HIVO
   *
   * @param string $uploadId
   *   The upload ID from HIVO
   * @param string $version
   *   The version of the file
   * @param string $userId
   *   The user ID
   * @param string $token
   *   The authentication token
   *
   * @return array|null
   *   Array containing file information or null on failure
   */
  protected function downloadFromHivo($uploadId, $version, $userId, $token) {
    // Get HIVO API URL from constant
    $hivoApiUrl = HivoConstants::HIVO_API_URL;
    
    // Build the API URL for assets download
    $apiUrl = $hivoApiUrl . '/api/drupal/assetsDownload';
    $params = http_build_query([
      'uploadId' => $uploadId,
      'version' => $version,
    ]);
    $fullUrl = $apiUrl . '?' . $params;

    // Use cURL for better header support
    $ch = curl_init();
    curl_setopt_array($ch, [
      CURLOPT_URL => $fullUrl,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_TIMEOUT => 30,
      CURLOPT_HTTPHEADER => [
        'x-auth-token: ' . $token,
        'x-user-id: ' . $userId,
        'Content-Type: application/json',
      ],
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($error) {
      throw new \Exception('cURL error: ' . $error);
    }

    if ($httpCode !== 200) {
      throw new \Exception('HTTP error: ' . $httpCode . ' - Response: ' . $response);
    }

    $data = json_decode($response, true);
    
    if (!$data || isset($data['error'])) {
      throw new \Exception('HIVO API error: ' . ($data['error'] ?? 'Unknown error'));
    }

    return $data;
  }

  /**
   * Download file content from URL
   *
   * @param string $url
   *   The file URL
   *
   * @return string|null
   *   File content or null on failure
   */
  protected function downloadFileContent($url) {
    $context = stream_context_create([
      'http' => [
        'timeout' => 60, // Longer timeout for file downloads
        'user_agent' => 'Drupal-HIVO-Connector/1.0',
      ],
    ]);

    $content = file_get_contents($url, false, $context);
    
    if ($content === false) {
      throw new \Exception('Failed to download file from URL: ' . $url);
    }

    return $content;
  }

  /**
   * Save file content to Drupal
   *
   * @param string $fileContent
   *   The file content
   * @param string $filename
   *   The filename
   *
   * @return array
   *   Save result information
   */
  protected function saveToDrupal($fileContent, $filename, $hivoUpload) {
    $destination = 'public://hivo_media/';
    
    // Check if directory exists and is writable
    $directory_created = $this->fileSystem->prepareDirectory($destination, FileSystemInterface::CREATE_DIRECTORY);
    
    if (!$directory_created) {
      throw new \Exception('Failed to create upload directory');
    }

    // Sanitize filename to prevent directory traversal and special characters
    $filename = basename($filename);
    $filename = preg_replace('/[^a-zA-Z0-9\-\_\.]/', '_', $filename);

    // Ensure filename is not empty after sanitization
    if (empty($filename)) {
      $filename = 'upload_' . time();
    }

    // Detect MIME type from file content
    $finfo = new \finfo(FILEINFO_MIME_TYPE);
    $mimeType = $finfo->buffer($fileContent);
    
    // Add extension if missing
    $filename = $this->addFileExtension($filename, $mimeType);
    
    $filepath = $destination . $filename;

    // TODO: Only insert if media with HIVO upload id does not exist
    // If not, update the media with the new file

    // Save file to Drupal using Drupal service
    $file = $this->fileRepository->writeData($fileContent, $filepath, FileSystemInterface::EXISTS_RENAME);
    
    if (!$file) {
      throw new \Exception('Failed to save file to Drupal');
    }

    // Set file as permanent
    $file->setPermanent();
    $file->save();

    // Detect media type
    $media_info = $this->detectMediaType($file->getFilename(), $file->getMimeType());
    
    // Create media entity
    $media = Media::create([
      'bundle' => $media_info['bundle'],
      'name' => $file->getFilename(),
      'uid' => \Drupal::currentUser()->id(),
      'status' => 1,
      $media_info['field'] => [
        'target_id' => $file->id(),
      ],
    ]);

    // Save Hivo data to media
    $media->set('hivo_data',json_encode([
      '_id' => $hivoUpload['uploadId'],
      'version' => $hivoUpload['version'],
      'lastSynced' => time(),
    ]));
    
    $media->save();

    return [
      'message' => 'Media upload successful!',
      'mid' => $media->id(),
      'fid' => $file->id(),
      'media_type' => $media_info['type'],
      'bundle' => $media_info['bundle'],
      'uri' => $file->getFileUri(),
      'filename' => $file->getFilename(),
      'mime_type' => $file->getMimeType(),
    ];
  }

  /**
   * Add file extension based on MIME type if missing
   *
   * @param string $filename
   *   The filename
   * @param string $mimeType
   *   The MIME type
   *
   * @return string
   *   Filename with proper extension
   */
  protected function addFileExtension($filename, $mimeType) {
    // Check if filename already has an extension
    $hasExtension = preg_match('/\.[a-zA-Z0-9]+$/', $filename);
    
    if ($hasExtension) {
      return $filename;
    }

    // Map common MIME types to file extensions
    $mimeToExtension = [
      // Images
      'image/jpeg' => '.jpg',
      'image/jpg' => '.jpg',
      'image/png' => '.png',
      'image/gif' => '.gif',
      'image/webp' => '.webp',
      'image/svg+xml' => '.svg',
      'image/bmp' => '.bmp',
      'image/tiff' => '.tiff',
      'image/tif' => '.tif',
      
      // Documents
      'application/pdf' => '.pdf',
      'text/plain' => '.txt',
      'text/html' => '.html',
      'text/css' => '.css',
      'text/javascript' => '.js',
      'application/javascript' => '.js',
      'application/json' => '.json',
      'application/xml' => '.xml',
      'application/zip' => '.zip',
      'application/x-zip-compressed' => '.zip',
      
      // Office documents
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => '.docx',
      'application/msword' => '.doc',
      'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' => '.xlsx',
      'application/vnd.ms-excel' => '.xls',
      'application/vnd.openxmlformats-officedocument.presentationml.presentation' => '.pptx',
      'application/vnd.ms-powerpoint' => '.ppt',
      
      // Videos
      'video/mp4' => '.mp4',
      'video/avi' => '.avi',
      'video/mov' => '.mov',
      'video/wmv' => '.wmv',
      'video/flv' => '.flv',
      'video/webm' => '.webm',
      
      // Audio
      'audio/mp3' => '.mp3',
      'audio/wav' => '.wav',
      'audio/ogg' => '.ogg',
      'audio/m4a' => '.m4a',
      'audio/aac' => '.aac',
    ];

    $extension = $mimeToExtension[$mimeType] ?? '';
    
    if ($extension) {
      return $filename . $extension;
    }

    // If no extension found, return original filename
    return $filename;
  }
}
