<?php

namespace Drupal\hold_my_draft\Controller;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityRepository;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Render\RendererInterface;
use Drupal\diff\Controller\NodeRevisionController;
use Drupal\diff\DiffEntityComparison;
use Drupal\diff\DiffLayoutManager;
use Drupal\node\Controller\NodeController as DefaultNodeController;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Drupal\hold_my_draft\Utilities;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class NodeController.
 *
 * @package Drupal\hold_my_draft\Controller
 */
class NodeController extends DefaultNodeController {


  /**
   * The draft-hold utilities service.
   *
   * @var \Drupal\hold_my_draft\Utilities
   */
  protected $utilities;

  /**
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected $moduleHandler;

  /**
   * The diff module's entity comparison which we conditionally inject.
   *
   * @var \Drupal\diff\DiffEntityComparison
   */
  protected $entityComparison;

  /**
   * The diff module's diff layout manager which we conditionally inject.
   *
   * @var \Drupal\diff\DiffLayoutManager
   */
  protected $layoutManager;

  /**
   * @var
   */
  protected $requestStack;

  /**
   * NodeController constructor.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   Date formatter for parent.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   Renderer for parent.
   * @param \Drupal\Core\Entity\EntityRepository $entityRepository
   *   Core entity repository.
   * @param \Drupal\hold_my_draft\Utilities $utilities
   *   Draft-hold utilities service.
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   The core module handler service.
   * @param \Drupal\diff\DiffEntityComparison $entityComparison
   *   The diff modules' entity comparison.
   * @param \Drupal\diff\DiffLayoutManager $layoutManager
   *   The diff module's layout manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The core symfony request service.
   */
  public function __construct(
    DateFormatterInterface $date_formatter,
    RendererInterface $renderer,
    EntityRepository $entityRepository,
    Utilities $utilities,
    ModuleHandler $moduleHandler,
    DiffEntityComparison $entityComparison = NULL,
    DiffLayoutManager $layoutManager = NULL,
    RequestStack $requestStack = NULL
  ) {
    parent::__construct($date_formatter, $renderer, $entityRepository);
    $this->utilities = $utilities;
    $this->moduleHandler = $moduleHandler;
    $this->entityComparison = $entityComparison;
    $this->layoutManager = $layoutManager;
    $this->requestStack = $requestStack;
  }

  /**
   * Get services from the global services container.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The services container.
   *
   * @return \Drupal\node\Controller\NodeController|\Drupal\hold_my_draft\Controller\NodeController
   *   The controller.
   */
  public static function create(ContainerInterface $container) {
    $module_handler = $container->get('module_handler');
    if ($module_handler->moduleExists('diff')) {
      return new static(
        $container->get('date.formatter'),
        $container->get('renderer'),
        $container->get('entity.repository'),
        $container->get('hold_my_draft.utilities'),
        $module_handler,
        $container->get('diff.entity_comparison'),
        $container->get('plugin.manager.diff.layout'),
        $container->get('request_stack'),
      );
    }
    else {
      return new static(
        $container->get('date.formatter'),
        $container->get('renderer'),
        $container->get('entity.repository'),
        $container->get('hold_my_draft.utilities'),
        $module_handler,
      );
    }

  }

  /**
   * Generates an overview table of older revisions of a node.
   *
   * This allows an added "Clone and edit" button for the current revision.
   * By default there are no actions available on the revision screen
   * for the current revision.
   *
   * @param \Drupal\node\NodeInterface $node
   *   A node object.
   *
   * @return array
   *   Build array.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function revisionOverview(NodeInterface $node) {
    // The diff module overrides the parent/ If it's installed we want to
    // modify their version of the screen, not the core one.
    $diff_installed = $this->moduleHandler->moduleExists('diff');
    if ($diff_installed) {
      // Because we only need these conditionally for the constructor we
      // conditionally dependency inject this in.
      $diffRevision = new NodeRevisionController($this->entityComparison, $this->layoutManager, $this->requestStack);
      $build = $diffRevision->revisionOverview($node);
    }
    else {
      // Get the build array from base controller method.
      $build = parent::revisionOverview($node);
    }

    $nid = $node->id();

    // We need to prevent multiple draft-holds from happening at once.
    // We also need to check this before draft-hold-ability because when a
    // draft-hold is first kicked off, it's not in a draft-holdable state.
    $draftHold = $this->utilities->getDraftHoldInfo($node);
    if ($this->utilities->isInProgress($draftHold)) {
      // Add a message.
      $this->utilities->generateMessage($draftHold, $node);
      return $build;
    }
    // Before proceeding, we should make sure that this ought to be draft-held.
    if (!$this->utilities->isDraftHoldable($node)) {
      return $build;
    }
    $vid = $this->utilities->getDefaultRevisionId($node);

    // Make sure the build array is what we expect.
    // The diff module removes it from rows.
    if ($build['node_revisions_table']) {
      $rows = [];
      if ($diff_installed) {
        $raw_rows = $build['node_revisions_table'];
        foreach ($raw_rows as $raw_row => $values) {
          if (is_numeric($raw_row)) {
            // It's a revision row from the Diff module, add it for modification.
            $rows[$raw_row] = $values;
          }
        }
      }
      elseif ($build['node_revisions_table']['#rows']) {
        $rows = $build['node_revisions_table']['#rows'];
      }
      else {
        // We don't know how to modify this version of the form, leave it be.
        return $build;
      }


      foreach ($rows as $row) {
        // In the parent controller, only the current revision gets a class.
        if (isset($row['class'])) {
          if (in_array('revision-current', $row['class'])) {
            $row['data'][1]['data']['#links'] = ['hotfix' => $this->addDraftHoldButton($nid, $vid)];
            $row['data'][1]['data']['#type'] = 'operations';
          }
        }
        // The diff module formats this a little differently. Check again.
        elseif (isset($row['#attributes']['class'])) {
          if (in_array('revision-current', $row['#attributes']['class'])) {
            $row['operations']['#links'] = ['hotfix' => $this->addDraftHoldButton($nid, $vid)];
            $row['operations']['#theme'] = 'links__dropbutton__operations';
            $row['operations']['#pre_render'][] = [
              0 => 'Drupal\Core\Render\Element\Operations',
              1 => 'preRenderDropbutton'
            ];
            unset($row['operations']['#markup']);
            unset($row['operations']['#prefix']);
            unset($row['operations']['#suffix']);
          }
        }
        $newrows[] = $row;
      }
      if (isset($newrows) && $build['node_revisions_table']['#rows']) {
        $build['node_revisions_table']['#rows'] = $newrows;
      }
      elseif ($diff_installed) {
        foreach ($newrows as $newrow => $values) {
          $build['node_revisions_table'][$newrow] = $values;
        }
      }
    }

    return $build;
  }

  /**
   * Create a draft-hold button link.
   *
   * @param int $nid
   *   The node id of the current page.
   * @param int $vid
   *   The revision id to be cloned and edited.
   *
   * @return array
   *   A link to be added to the render array.
   */
  protected function addDraftHoldButton(int $nid, int $vid) {
    $draftHold = [
      'title' => $this->t('Clone and edit'),
      'url' => Url::fromRoute('hold_my_draft.draft_hold_start', ['node' => $nid, 'node_revision' => $vid]),
    ];

    return $draftHold;
  }

}
