<?php

namespace Drupal\honeycronpot\Config;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryOverrideInterface;
use Drupal\Core\Config\StorageInterface;
use Drupal\honeycronpot\Service\HoneycronpotElementNameGenerator;

/**
 * Overrides honeypot.settings element_name dynamically.
 */
class HoneycronpotConfigOverride implements ConfigFactoryOverrideInterface {

  /**
   * Constructs a HoneycronpotConfigOverride object.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend interface.
   * @param \Drupal\Core\Config\StorageInterface $config_storage
   *   The config storage interface.
   * @param \Drupal\honeycronpot\Service\HoneycronpotElementNameGenerator $elementNameGenerator
   *   The element name generator service.
   */
  public function __construct(
    protected readonly CacheBackendInterface $cache,
    protected readonly StorageInterface $config_storage,
    protected readonly HoneycronpotElementNameGenerator $elementNameGenerator,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function loadOverrides($names): array {
    $overrides = [];
    $raw = $this->config_storage->read('honeypot.settings') ?: [];
    $mode = $raw['element_name_mode'] ?? 'cache';
    if ($mode !== 'cache') {
      return $overrides;
    }

    if (in_array('honeypot.settings', $names)) {
      $cache_id = 'honeycronpot.element_name';
      $cached = $this->cache->get($cache_id);

      $ttl = isset($raw['cache_ttl']) ? (int) $raw['cache_ttl'] : 86400;

      if (!$cached) {
        $element_name = $this->elementNameGenerator->generate();

        // Cache the generated element name for 24 hours (86400 seconds).
        $this->cache->set($cache_id, $element_name, time() + $ttl);
      }
      else {
        $element_name = $cached->data;
      }

      $overrides['honeypot.settings'] = [
        'element_name' => $element_name,
      ];
    }

    return $overrides;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheSuffix() {
    return 'HoneycronpotOverride';
  }

  /**
   * {@inheritdoc}
   */
  public function createConfigObject($name, $collection = StorageInterface::DEFAULT_COLLECTION) {
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheableConfig() {
    return ['honeypot.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheableMetadata($name) {
    $metadata = new CacheableMetadata();
    $metadata->setCacheTags(['config:honeycronpot.element_name']);

    return $metadata;

  }

}
