<?php

namespace Drupal\honeycronpot\Service;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Config\StorageInterface;

/**
 * Service for generating valid and unique honeypot element names.
 */
class HoneycronpotElementNameGenerator implements HoneycronpotElementNameGeneratorInterface {

  /**
   * Common base names for honeypot elements.
   */
  protected array $baseNames = [
    'name', 'email', 'homepage', 'link', 'contact', 'user', 'address', 'phone', 'mobile', 'url', 'website',
    'location', 'firstname', 'lastname', 'company', 'organization', 'zipcode', 'city', 'street', 'username',
    'message', 'comment', 'feedback', 'text', 'info', 'description', 'title', 'subject', 'formfield', 'data',
  ];

  /**
   * Common prefixes to prepend to base names.
   */
  protected array $prefixes = [
    'customer_', 'client_', 'account_', 'member_', 'contact_', 'profile_', 'person_', 'applicant_', 'registrant_',
    'participant_', 'subscriber_', 'lead_', 'visitor_', 'guest_', 'requester_', 'submitter_', 'respondent_',
    'enquirer_', 'follower_', 'donor_', 'supporter_', 'attendee_', 'memberinfo_', 'signup_', 'joiner_', 'volunteer_',
    'buyer_', 'userinput_', 'formuser_', 'interested_',
  ];

  /**
   * Constructs the generator service.
   */
  public function __construct(
    protected readonly StorageInterface $configStorage,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function generate(): string {
    $existing = $this->getExistingElementNames();

    do {
      $random_base = $this->baseNames[array_rand($this->baseNames)];
      $random_prefix = $this->prefixes[array_rand($this->prefixes)];
      $element_name = $random_prefix . $random_base;

      $is_valid = preg_match('/^[-_a-zA-Z0-9]+$/', $element_name)
        && preg_match('/^[a-zA-Z]/', $element_name)
        && !in_array($random_base, ['name', 'pass', 'website'])
        && !in_array($element_name, $existing);
    } while (!$is_valid);

    return $element_name;
  }

  /**
   * Collects all existing element names from webform configurations.
   */
  protected function getExistingElementNames(): array {
    $names = [];

    $configs = $this->configStorage->listAll('webform.webform.');
    foreach ($configs as $config_name) {
      $data = $this->configStorage->read($config_name);
      if (!empty($data['elements'])) {
        $elements = Yaml::decode($data['elements']);
        if (is_array($elements)) {
          $names = array_merge($names, $this->extractElementKeys($elements));
        }
      }
    }

    return array_unique($names);
  }

  /**
   * Recursively extract all element keys from a Webform elements array.
   *
   * @param array $elements
   *   The decoded elements array from the Webform config.
   *
   * @return array
   *   All element keys (flattened).
   */
  protected function extractElementKeys(array $elements): array {
    $keys = [];

    foreach ($elements as $key => $value) {
      if (str_starts_with($key, '#')) {
        continue;
      }

      $keys[] = $key;
      if (is_array($value)) {
        // Only consider it an element if it has a '#type' property.
        if (isset($value['#type'])) {
          $keys[] = $key;
        }
        // Dive deeper if this element has nested elements.
        $keys = array_merge($keys, $this->extractElementKeys($value));
      }
    }

    return $keys;
  }

}
