/**
 * @file
 * Attaches the behaviors for the Hover Card module using Tippy.js.
 */

(function (Drupal, drupalSettings, tippy) {
  /**
   * Generates & displays the required values for the Hover Card.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.hoverCard = {
    attach(context) {
      // Get configuration from Drupal settings.
      const config = drupalSettings.hoverCard || {};
      const selector =
        config.selector ||
        'a.username, span[rel="schema:author"] > a, span.node__author a';
      const basePath = config.basePath || '';

      // Find all user links that haven't been processed yet.
      const userLinks = context.querySelectorAll(selector);

      userLinks.forEach(function (link) {
        // Skip if already processed.
        if (link.hasAttribute('data-hover-card-processed')) {
          return;
        }

        // Mark as processed.
        link.setAttribute('data-hover-card-processed', 'true');

        // Extract user ID from the link href.
        const href = link.getAttribute('href');
        if (!href) {
          return;
        }

        // Extract user ID from URL (e.g., /user/123).
        const userIdMatch = href.match(/\/user\/(\d+)/);
        if (!userIdMatch) {
          return;
        }

        const userId = userIdMatch[1];

        // Create Tippy instance with lazy loading.
        tippy(link, {
          content:
            '<div class="hover-card hover-card--loading">Loading...</div>',
          allowHTML: true,
          interactive: true,
          theme: 'hover-card',
          placement: 'right',
          maxWidth: 350,
          trigger: 'mouseenter focus',
          delay: [200, 0],
          animation: 'fade',
          onShow(instance) {
            // Check if content already loaded.
            if (instance._contentLoaded) {
              return;
            }

            // Fetch user data.
            const url = `${basePath}/hover-card/${userId}`;

            fetch(url)
              .then((response) => {
                if (!response.ok) {
                  throw new Error('Network response was not ok');
                }
                return response.text();
              })
              .then((html) => {
                instance.setContent(html);
                instance._contentLoaded = true;
              })
              .catch((error) => {
                console.error('Hover Card Error:', error);
                instance.setContent(
                  '<div class="hover-card hover-card--error">Unable' +
                    ' to load user information.</div>',
                );
              });
          },
        });
      });
    },
  };
})(Drupal, drupalSettings, window.tippy);
