<?php

namespace Drupal\hover_card\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Hover Card settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Constructs a SettingsForm object.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'hover_card_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['hover_card.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('hover_card.settings');

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Configure which user information fields should be displayed in the hover card. The user name is always displayed.') . '</p>',
    ];

    $form['display_fields'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Display Settings'),
      '#description' => $this->t('Select which fields to display in the hover card.'),
    ];

    $form['display_fields']['display_picture'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display user picture'),
      '#config_target' => 'hover_card.settings:display_picture',
      '#description' => $this->t('Show the user profile picture if available.'),
    ];

    $form['display_fields']['display_email'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display email address'),
      '#config_target' => 'hover_card.settings:display_email',
      '#description' => $this->t('Show the user email address.'),
    ];

    $form['display_fields']['display_roles'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display user roles'),
      '#config_target' => 'hover_card.settings:display_roles',
      '#description' => $this->t('Show the user roles.'),
    ];

    $form['display_fields']['display_member_since'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display member since date'),
      '#config_target' => 'hover_card.settings:display_member_since',
      '#description' => $this->t('Show when the user account was created.'),
    ];

    $form['display_fields']['display_last_access'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display last access date'),
      '#config_target' => 'hover_card.settings:display_last_access',
      '#description' => $this->t('Show when the user last accessed the site.'),
    ];

    $form['display_fields']['display_bio'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display user bio/description'),
      '#config_target' => 'hover_card.settings:display_bio',
      '#description' => $this->t('Show the user bio or description field if available (field_description).'),
    ];

    // Get custom user fields.
    $user_fields = $this->getCustomUserFields();
    $enabled_fields = $config->get('user_fields') ?? [];

    if (!empty($user_fields)) {
      $form['custom_fields'] = [
        '#type' => 'details',
        '#title' => $this->t('Custom User Fields'),
        '#description' => $this->t('Select which custom fields added to the user entity should be displayed in the hover card.'),
        '#open' => TRUE,
      ];

      foreach ($user_fields as $field_name => $field_definition) {
        $form['custom_fields']['user_field_' . $field_name] = [
          '#type' => 'checkbox',
          '#title' => $field_definition['label'],
          '#default_value' => in_array($field_name, $enabled_fields, TRUE),
          '#description' => $this->t('Type: @type', ['@type' => $field_definition['type']]),
        ];
      }
    }

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
    ];

    $form['advanced']['selector'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CSS Selector'),
      '#config_target' => 'hover_card.settings:selector',
      '#description' => $this->t('The CSS selector to target user links for hover cards. Default: <code>a.username, span[rel="schema:author"] > a, span.node__author a</code>'),
      '#size' => 60,
      '#maxlength' => 255,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Standard fields (display_picture, display_email, etc.) are automatically
    // saved by the parent class via #config_target.
    // We only need to manually save custom user fields.
    $user_fields = $this->getCustomUserFields();
    if (!empty($user_fields)) {
      $enabled_fields = [];
      foreach ($user_fields as $field_name => $field_definition) {
        $form_value = $form_state->getValue('user_field_' . $field_name);
        if ($form_value) {
          $enabled_fields[] = $field_name;
        }
      }
      $this->config('hover_card.settings')
        ->set('user_fields', $enabled_fields)
        ->save();
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets custom user fields (non-base fields).
   *
   * @return array
   *   An array of custom field definitions with field_name as key and
   *   an array containing 'label' and 'type' as value.
   */
  protected function getCustomUserFields(): array {
    $fields = [];
    $field_definitions = $this->entityFieldManager->getFieldDefinitions('user', 'user');

    // Base fields to exclude.
    $base_fields = [
      'uid', 'uuid', 'langcode', 'preferred_langcode', 'preferred_admin_langcode',
      'name', 'pass', 'mail', 'timezone', 'status', 'created', 'changed',
      'access', 'login', 'init', 'roles', 'default_langcode', 'user_picture',
    ];

    foreach ($field_definitions as $field_name => $field_definition) {
      // Skip base fields and computed fields.
      if (in_array($field_name, $base_fields) || $field_definition->isComputed()) {
        continue;
      }

      // Only include fields that can be displayed.
      if ($field_definition->isDisplayConfigurable('view')) {
        $fields[$field_name] = [
          'label' => $field_definition->getLabel(),
          'type' => $field_definition->getType(),
        ];
      }
    }

    return $fields;
  }

}
