<?php

namespace Drupal\hover_card\Service;

use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user\UserInterface;

/**
 * Service for preparing hover card user data.
 */
class HoverCardDataService {

  use StringTranslationTrait;

  /**
   * Constructs a HoverCardDataService object.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected DateFormatterInterface $dateFormatter,
  ) {}

  /**
   * Gets user data for hover card display.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user entity.
   *
   * @return array
   *   An array of user data based on configuration settings.
   */
  public function getUserData(UserInterface $user): array {
    $config = $this->configFactory->get('hover_card.settings');
    $data = [];

    // Name is always displayed.
    $data['name'] = Html::escape($user->getDisplayName());

    // Email.
    if ($config->get('display_email') && $user->getEmail()) {
      $data['mail'] = Html::escape($user->getEmail());
    }

    // Picture.
    if ($config->get('display_picture') && $user->hasField('user_picture') && !$user->get('user_picture')->isEmpty()) {
      $data['picture'] = $user->get('user_picture')->view('thumbnail');
    }

    // Roles.
    if ($config->get('display_roles')) {
      $roles = [];
      foreach ($user->getRoles() as $role) {
        $roles[] = Html::escape($role);
      }
      $data['roles'] = implode(', ', $roles);
    }

    // Member since.
    if ($config->get('display_member_since') && $user->getCreatedTime()) {
      $data['member_since'] = $this->dateFormatter->format($user->getCreatedTime(), 'medium');
    }

    // Last access.
    if ($config->get('display_last_access') && $user->getLastAccessedTime()) {
      $last_access = $user->getLastAccessedTime();
      if ($last_access > 0) {
        $data['last_access'] = $this->dateFormatter->format($last_access, 'medium');
      }
    }

    // Bio/Description from field_description if it exists.
    if ($config->get('display_bio') && $user->hasField('field_description') && !$user->get('field_description')->isEmpty()) {
      $bio_value = $user->get('field_description')->value;
      // Limit bio length for hover card display.
      $data['bio'] = Html::escape(mb_substr($bio_value, 0, 200));
      if (mb_strlen($bio_value) > 200) {
        $data['bio'] .= '...';
      }
    }

    // Handle custom user fields.
    $enabled_fields = $config->get('user_fields') ?? [];
    foreach ($enabled_fields as $field_name) {
      if ($user->hasField($field_name) && !$user->get($field_name)->isEmpty()) {
        $field_item = $user->get($field_name);
        $field_definition = $field_item->getFieldDefinition();
        $field_type = $field_definition->getType();

        // Handle different field types.
        $data['custom_fields'][$field_name] = [
          'label' => $field_definition->getLabel(),
          'value' => $this->formatFieldValue($field_item, $field_type),
        ];
      }
    }

    return $data;
  }

  /**
   * Gets the CSS selector for hover card targets.
   *
   * @return string
   *   The CSS selector string.
   */
  public function getSelector(): string {
    $config = $this->configFactory->get('hover_card.settings');
    return $config->get('selector') ?? 'a.username, span[rel="schema:author"] > a, span.node__author a';
  }

  /**
   * Formats a field value for display in the hover card.
   *
   * @param mixed $field_item
   *   The field item to format.
   * @param string $field_type
   *   The field type.
   *
   * @return string|array
   *   The formatted field value.
   */
  protected function formatFieldValue($field_item, string $field_type) {
    // Handle common field types.
    switch ($field_type) {
      case 'string':
      case 'string_long':
        // Plain text fields - escape HTML.
        $value = $field_item->value;
        if (mb_strlen($value) > 200) {
          return Html::escape(mb_substr($value, 0, 200)) . '...';
        }
        return Html::escape($value);

      case 'text':
      case 'text_long':
      case 'text_with_summary':
        // Formatted text fields - use processed text or strip tags.
        if (isset($field_item->processed)) {
          // Use processed text (with filters applied).
          $value = strip_tags($field_item->processed);
        }
        else {
          // Fallback to stripping tags from raw value.
          $value = strip_tags($field_item->value);
        }
        // Limit text length.
        if (mb_strlen($value) > 200) {
          return mb_substr($value, 0, 200) . '...';
        }
        return $value;

      case 'integer':
      case 'decimal':
      case 'float':
        return Html::escape((string) $field_item->value);

      case 'boolean':
        return $field_item->value ? $this->t('Yes') : $this->t('No');

      case 'datetime':
      case 'timestamp':
        if (isset($field_item->value)) {
          return $this->dateFormatter->format(strtotime($field_item->value), 'medium');
        }
        return '';

      case 'email':
        return Html::escape($field_item->value);

      case 'telephone':
        return Html::escape($field_item->value);

      case 'link':
        if ($field_item->uri) {
          return Html::escape($field_item->title ?: $field_item->uri);
        }
        return '';

      case 'entity_reference':
      case 'image':
      case 'file':
        // For complex fields, render them using the view builder.
        return $field_item->view(['label' => 'hidden', 'settings' => ['image_style' => 'thumbnail']]);

      case 'list_string':
      case 'list_integer':
      case 'list_float':
        // Get the allowed value label.
        $allowed_values = $field_item->getFieldDefinition()->getSetting('allowed_values');
        $value = $field_item->value;
        return Html::escape($allowed_values[$value] ?? $value);

      default:
        // For unknown types, try to get a string representation.
        if (isset($field_item->value)) {
          $value = (string) $field_item->value;
          if (mb_strlen($value) > 200) {
            return Html::escape(mb_substr($value, 0, 200)) . '...';
          }
          return Html::escape($value);
        }
        return '';
    }
  }

}
