<?php

namespace Drupal\Tests\hover_card\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Functional tests for Hover Card settings form.
 *
 * @group hover_card
 */
class SettingsFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['hover_card', 'user'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with permission to administer hover card settings.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer hover card settings',
    ]);
  }

  /**
   * Tests the settings form access.
   */
  public function testSettingsFormAccess() {
    // Anonymous users should not have access.
    $this->drupalGet('/admin/config/people/hover-card');
    $this->assertSession()->statusCodeEquals(403);

    // Admin user should have access.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/people/hover-card');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Configure which user information fields should be displayed in the hover card');
  }

  /**
   * Tests the settings form submission.
   */
  public function testSettingsFormSubmission() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/people/hover-card');

    // Submit the form with some settings enabled.
    $edit = [
      'display_picture' => TRUE,
      'display_email' => TRUE,
      'display_roles' => TRUE,
      'display_member_since' => FALSE,
      'display_last_access' => FALSE,
      'display_bio' => FALSE,
      'selector' => 'a.username',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved');

    // Verify the configuration was saved.
    $config = $this->config('hover_card.settings');
    $this->assertTrue($config->get('display_picture'));
    $this->assertTrue($config->get('display_email'));
    $this->assertTrue($config->get('display_roles'));
    $this->assertFalse($config->get('display_member_since'));
    $this->assertEquals('a.username', $config->get('selector'));
  }

  /**
   * Tests form fields are present.
   */
  public function testFormFields() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/people/hover-card');

    // Check that all expected form fields exist.
    $this->assertSession()->fieldExists('display_picture');
    $this->assertSession()->fieldExists('display_email');
    $this->assertSession()->fieldExists('display_roles');
    $this->assertSession()->fieldExists('display_member_since');
    $this->assertSession()->fieldExists('display_last_access');
    $this->assertSession()->fieldExists('display_bio');
    $this->assertSession()->fieldExists('selector');
  }

  /**
   * Tests that selector field is required.
   */
  public function testSelectorRequired() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/people/hover-card');

    // Try to submit with empty selector.
    $edit = [
      'selector' => '',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('CSS Selector field is required');
  }

}
