<?php

namespace Drupal\Tests\hover_card\Unit;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\hover_card\Service\HoverCardDataService;
use Drupal\Tests\UnitTestCase;
use Drupal\user\UserInterface;

/**
 * Unit tests for HoverCardDataService.
 *
 * @coversDefaultClass \Drupal\hover_card\Service\HoverCardDataService
 * @group hover_card
 */
class HoverCardDataServiceTest extends UnitTestCase {

  /**
   * The mocked config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The hover card data service.
   *
   * @var \Drupal\hover_card\Service\HoverCardDataService
   */
  protected $service;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);

    $this->service = new HoverCardDataService(
      $this->configFactory,
      $this->entityTypeManager,
      $this->createMock(DateFormatterInterface::class)
    );
  }

  /**
   * Tests getUserData method with minimal configuration.
   *
   * @covers ::getUserData
   */
  public function testGetUserDataMinimal() {
    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')
      ->willReturnMap([
        ['display_email', FALSE],
        ['display_picture', FALSE],
        ['display_roles', FALSE],
        ['display_member_since', FALSE],
        ['display_last_access', FALSE],
        ['display_bio', FALSE],
        ['user_fields', []],
      ]);

    $this->configFactory->method('get')
      ->with('hover_card.settings')
      ->willReturn($config);

    $user = $this->createMock(UserInterface::class);
    $user->method('getDisplayName')->willReturn('Test User');

    $data = $this->service->getUserData($user);

    $this->assertIsArray($data);
    $this->assertArrayHasKey('name', $data);
    $this->assertEquals('Test User', $data['name']);
    $this->assertArrayNotHasKey('mail', $data);
    $this->assertArrayNotHasKey('roles', $data);
  }

  /**
   * Tests getUserData method with email enabled.
   *
   * @covers ::getUserData
   */
  public function testGetUserDataWithEmail() {
    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')
      ->willReturnMap([
        ['display_email', TRUE],
        ['display_picture', FALSE],
        ['display_roles', FALSE],
        ['display_member_since', FALSE],
        ['display_last_access', FALSE],
        ['display_bio', FALSE],
        ['user_fields', []],
      ]);

    $this->configFactory->method('get')
      ->with('hover_card.settings')
      ->willReturn($config);

    $user = $this->createMock(UserInterface::class);
    $user->method('getDisplayName')->willReturn('Test User');
    $user->method('getEmail')->willReturn('test@example.com');

    $data = $this->service->getUserData($user);

    $this->assertArrayHasKey('mail', $data);
    $this->assertEquals('test@example.com', $data['mail']);
  }

  /**
   * Tests getUserData method with roles enabled.
   *
   * @covers ::getUserData
   */
  public function testGetUserDataWithRoles() {
    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')
      ->willReturnMap([
        ['display_email', FALSE],
        ['display_picture', FALSE],
        ['display_roles', TRUE],
        ['display_member_since', FALSE],
        ['display_last_access', FALSE],
        ['display_bio', FALSE],
        ['user_fields', []],
      ]);

    $this->configFactory->method('get')
      ->with('hover_card.settings')
      ->willReturn($config);

    $user = $this->createMock(UserInterface::class);
    $user->method('getDisplayName')->willReturn('Test User');
    $user->method('getRoles')->willReturn(['authenticated', 'editor']);

    $data = $this->service->getUserData($user);

    $this->assertArrayHasKey('roles', $data);
    $this->assertEquals('authenticated, editor', $data['roles']);
  }

  /**
   * Tests getSelector method.
   *
   * @covers ::getSelector
   */
  public function testGetSelector() {
    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')
      ->with('selector')
      ->willReturn('a.custom-selector');

    $this->configFactory->method('get')
      ->with('hover_card.settings')
      ->willReturn($config);

    $selector = $this->service->getSelector();
    $this->assertEquals('a.custom-selector', $selector);
  }

  /**
   * Tests getSelector method with default value.
   *
   * @covers ::getSelector
   */
  public function testGetSelectorDefault() {
    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')
      ->with('selector')
      ->willReturn(NULL);

    $this->configFactory->method('get')
      ->with('hover_card.settings')
      ->willReturn($config);

    $selector = $this->service->getSelector();
    $this->assertEquals('a.username, span[rel="schema:author"] > a, span.node__author a', $selector);
  }

}
