<?php

namespace Drupal\hover_speech\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines the configuration form for Hover Speech settings.
 */
class HoverSpeechSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   *
   * Returns the names of the editable configuration objects.
   */
  protected function getEditableConfigNames() {
    return ['hover_speech.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * Returns the form ID for this configuration form.
   */
  public function getFormId() {
    return 'hover_speech_settings_form';
  }

  /**
   * {@inheritdoc}
   *
   * Builds the Hover Speech settings form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The form definition array.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('hover_speech.settings');

    // Language mode.
    $form['language_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Speech language mode'),
      '#options' => [
        'auto' => $this->t('Auto detect'),
        'english' => $this->t('Force English'),
        'marathi' => $this->t('Force Marathi/Hindi'),
      ],
      '#default_value' => $config->get('language_mode') ?? 'auto',
    ];

    // Colors.
    $form['button_colors'] = [
      '#type' => 'details',
      '#title' => $this->t('Button Colors'),
      '#open' => TRUE,
    ];

    $form['button_colors']['background_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Background color'),
      '#default_value' => $config->get('background_color') ?? '#28a745',
    ];

    $form['button_colors']['text_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Text color'),
      '#default_value' => $config->get('text_color') ?? '#ffffff',
    ];

    $form['button_colors']['border_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Border color'),
      '#default_value' => $config->get('border_color') ?? '#1e7e34',
    ];

    $form['button_colors']['hover_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Hover background color'),
      '#default_value' => $config->get('hover_color') ?? '#218838',
    ];

    // Button shape.
    $form['button_shape'] = [
      '#type' => 'select',
      '#title' => $this->t('Button shape'),
      '#options' => [
        'square' => $this->t('Square'),
        'rounded' => $this->t('Rounded'),
        'circle' => $this->t('Circle'),
      ],
      '#default_value' => $config->get('button_shape') ?? 'rounded',
    ];

    // Position of button.
    $form['button_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Button position'),
      '#options' => [
        'top-left' => $this->t('Top Left'),
        'top-right' => $this->t('Top Right'),
        'bottom-left' => $this->t('Bottom Left'),
        'bottom-right' => $this->t('Bottom Right'),
      ],
      '#default_value' => $config->get('button_position') ?? 'bottom-right',
    ];

    // Button style settings.
    $form['button_style'] = [
      '#type' => 'details',
      '#title' => $this->t('Button style settings'),
      '#open' => TRUE,
    ];

    $form['button_style']['button_font_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Speech button font size (px)'),
      '#default_value' => $config->get('button_font_size') ?? 14,
      '#min' => 10,
      '#max' => 40,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * Submits the configuration form values.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('hover_speech.settings')
      ->set('language_mode', $form_state->getValue('language_mode'))
      ->set('background_color', $form_state->getValue('background_color'))
      ->set('text_color', $form_state->getValue('text_color'))
      ->set('border_color', $form_state->getValue('border_color'))
      ->set('hover_color', $form_state->getValue('hover_color'))
      ->set('button_shape', $form_state->getValue('button_shape'))
      ->set('button_position', $form_state->getValue('button_position'))
      ->set('button_font_size', $form_state->getValue('button_font_size'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
