<?php

namespace Drupal\htaccess\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\File\FileSystemInterface;

/**
 * Configure htaccess settings for this site.
 */
class HtaccessAdminSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'htaccess_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['htaccess.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('htaccess.settings');
    $module_handler = \Drupal::service('module_handler');

    // Editable field for the default .htaccess path, always visible.
    $form['default_htaccess_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default .htaccess file path'),
      '#default_value' => $config->get('default_htaccess_path') ?: 'core/assets/scaffold/files/htaccess',
      '#description' => $this->t('Specify the path from which the default .htaccess content is loaded.'),
      '#required' => TRUE,
    ];

    // Load content from the specified path.
    $default_htaccess = '';
    $default_htaccess_path = $form_state->getValue('default_htaccess_path') ?: ($config->get('default_htaccess_path') ?: 'core/assets/scaffold/files/htaccess');
    if (file_exists($default_htaccess_path) && is_readable($default_htaccess_path)) {
      $default_htaccess = file_get_contents($default_htaccess_path);
    } else {
      $default_htaccess = '# The specified path does not exist or is not readable. Please verify the path.';
    }

    // Read-only field to display the default content.
    $form['content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Default .htaccess content (read-only)'),
      '#default_value' => $default_htaccess,
      '#cols' => 60,
      '#rows' => 10,
      '#attributes' => ['readonly' => 'readonly'],
      '#description' => $this->t('This is the default .htaccess content loaded from the specified path.'),
    ];

    // Extra configurations field: editable for customizations.
    $form['configuraciones_extra'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Extra configurations'),
      '#default_value' => $config->get('configuraciones_extra'),
      '#cols' => 60,
      '#rows' => 10,
      '#description' => $this->t('Add your custom configurations here. They will be combined with the default content on save.'),
    ];

    // Checkbox to automatically replace .htaccess.
    $form['reemplazar_automaticamente'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Automatically replace .htaccess'),
      '#default_value' => $config->get('reemplazar_automaticamente'),
      '#description' => $this->t('If checked, the .htaccess file will be updated with the combined content every time cron runs.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $default_htaccess_path = $form_state->getValue('default_htaccess_path');
    if (!file_exists($default_htaccess_path) || !is_readable($default_htaccess_path)) {
      $form_state->setErrorByName('default_htaccess_path', $this->t('The specified path is incorrect or the file cannot be read. Verify the path and try again.'));
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('htaccess.settings');
    $default_htaccess_path = $form_state->getValue('default_htaccess_path');
    $extra_content = $form_state->getValue('configuraciones_extra');
    $reemplazar_automaticamente = $form_state->getValue('reemplazar_automaticamente');

    // Save configurations.
    $config
      ->set('default_htaccess_path', $default_htaccess_path)
      ->set('configuraciones_extra', $extra_content)
      ->set('reemplazar_automaticamente', $reemplazar_automaticamente)
      ->save();

    // Load the default content from the specified path.
    $default_content = file_get_contents($default_htaccess_path); // Already validated in validateForm.

    // Combine with extra configurations.
    $full_content = $default_content . "\n\n" . $extra_content;
    $file_system = \Drupal::service('file_system');
    $htaccess_destination = DRUPAL_ROOT . '/.htaccess';
    if (is_writable($htaccess_destination) || (!file_exists($htaccess_destination) && is_writable(dirname($htaccess_destination)))) {
      $file_system->saveData($full_content, $htaccess_destination, FileSystemInterface::EXISTS_REPLACE);
      $this->messenger()->addMessage($this->t('.htaccess file updated at @path.', ['@path' => $htaccess_destination]));
    } else {
      $this->messenger()->addError($this->t('Could not write to .htaccess at @path. Check permissions.', ['@path' => $htaccess_destination]));
    }

    Cache::invalidateTags(['htaccess']);
    parent::submitForm($form, $form_state);
  }
}