<?php

namespace Drupal\html_lang_override\EventSubscriber;

/**
 * @file
 * Contains \Drupal\html_lang_override\EventSubscriber\HtmlLangSubscriber.
 */

use Drupal\Core\Render\HtmlResponse;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\html_lang_override\HtmlLangManager;

/**
 * Listens to the kernel.response event.
 */
class HtmlLangSubscriber implements EventSubscriberInterface {

  /**
   * The language code override service.
   *
   * @var \Drupal\html_lang_override\HtmlLangManager
   */
  protected $langManager;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Constructs a new HtmlLangSubscriber object.
   *
   * @param \Drupal\html_lang_override\HtmlLangManager $langManager
   *   The language code override service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   */
  public function __construct(HtmlLangManager $langManager, RouteMatchInterface $route_match) {
    $this->langManager = $langManager;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [KernelEvents::RESPONSE => 'onRespond'];
  }

  /**
   * Responds to the kernel.response event.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onRespond(ResponseEvent $event) {

    $lang = NULL;
    $node = $this->routeMatch->getParameter('node');
    // Get language code for Node ID if available.
    if ($node && $node->hasField('nid')) {
      $node_lang = $this->langManager->getLangForNode($node->id());
      if ($node_lang) {
        $lang = $node_lang;
      }
    }
    // Get language code for Request Path if available.
    if ($lang === NULL) {
      $path = $event->getRequest()->getPathInfo();
      $lang = $this->langManager->getLangForPath($path);
    }
    // Get default language code if not customized.
    if ($lang === NULL) {
      $lang = $this->langManager->getDefaultLang();
    }

    $response = $event->getResponse();
    if (!$response instanceof HtmlResponse) {
      return;
    }

    $content = $response->getContent();
    $content = preg_replace('/<html(.*?)lang=".*?"/', "<html$1lang=\"$lang\"", $content);
    $response->setContent($content);
  }

}
