<?php

namespace Drupal\html_lang_override;

/**
 * @file
 * Contains \Drupal\html_lang_override\HtmlLangManager.
 */

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Messenger\Messenger;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Component\Utility\Html;

/**
 * Class HtmlLangManager provides various methods for language code overrides.
 *
 * @package Drupal\html_lang_override
 */
class HtmlLangManager {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\Messenger
   */
  protected $messenger;

  /**
   * Constructs a new HtmlLangManager object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Messenger\Messenger $messenger
   *   The messenger service.
   */
  public function __construct(Connection $database, ConfigFactoryInterface $config_factory, LanguageManagerInterface $language_manager, TimeInterface $time, Messenger $messenger) {
    $this->database = $database;
    $this->config = $config_factory->getEditable('html_lang_override.settings');
    $this->languageManager = $language_manager;
    $this->time = $time;
    $this->messenger = $messenger;
  }

  /**
   * Gets the default language code.
   *
   * @return string
   *   The default language code.
   */
  public function getDefaultLang() {
    if ($this->config->get('override_html_lang')) {
      $custom_lang = $this->config->get('default_lang');
      if (!empty($custom_lang)) {
        return $custom_lang;
      }
    }
    return $this->languageManager->getDefaultLanguage()->getId();
  }

  /**
   * Sets the default language code.
   *
   * @param string $nid
   *   Get language code for node.
   */
  public function getLangForNode($nid) {

    if (!$nid) {
      return NULL;
    }
    try {
      return $this->database->select('html_lang_override_node', 'hl')
        ->fields('hl', ['lang'])
        ->condition('nid', $nid)
        ->execute()
        ->fetchField() ?? NULL;
    }
    catch (\Exception $e) {
      // No need to show error message to user.
    }
    return NULL;
  }

  /**
   * Saves the language code for a node.
   *
   * @param int $nid
   *   The node ID.
   * @param string $lang
   *   The language code.
   */
  public function saveLangForNode($nid, $lang) {
    $lang = Html::escape(trim($lang));
    if (empty($nid) || empty($lang)) {
      return;
    }
    try {
      $this->database->merge('html_lang_override_node')
        ->key(['nid' => $nid])
        ->fields([
          'lang' => $lang,
          'changed' => $this->time->getRequestTime(),
          'created' => $this->time->getRequestTime(),
        ])
        ->execute();
    }
    catch (\Exception $e) {
      $this->messenger->addError(new TranslatableMarkup('An error occurred while saving the language code for the node. Please try again later.'));
    }
  }

  /**
   * Gets the language code for a specific path.
   *
   * @param string $path
   *   The path to check.
   *
   * @return string
   *   The language code.
   */
  public function getLangForPath($path) {
    $all_path_overrides = (string) $this->config->get('route_path_override');
    if ($all_path_overrides) {
      $path_array = explode("\n", $all_path_overrides);
      foreach ($path_array as $item) {
        [$route, $language] = explode('|', trim($item));
        if (trim($path) == trim($route)) {
          return trim($language);
        }
      }
    }
    return NULL;
  }

  /**
   * Deletes the language code for a node.
   *
   * @param int $nid
   *   The node ID.
   */
  public function deleteLangForNode($nid) {
    try {
      $this->database->delete('html_lang_override_node')
        ->condition('nid', $nid)
        ->execute();
    }
    catch (\Exception $e) {
      $this->messenger->addError(new TranslatableMarkup('An error occurred while deleting the language code for the node. Please try again later.'));
    }
  }

}
