<?php

namespace Drupal\html_lang_override\Form;

/**
 * @file
 * Contains \Drupal\html_lang_override\Form\HtmlLangOverrideForm.
 */

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form to configure Global language code override settings.
 *
 * @package Drupal\html_lang_override\Form
 */
class HtmlLangConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['html_lang_override.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'html_lang_override_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('html_lang_override.settings');

    $form['override_html_lang'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Override Default HTML lang attribute Globally.'),
      '#default_value' => $config->get('override_html_lang'),
      '#description' => $this->t('Check this to override the HTML lang attribute for all pages.'),
      '#access' => $this->currentUser()->hasPermission('Administer HTML lang override settings'),
    ];

    $form['default_lang'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default Language Code'),
      '#default_value' => $config->get('default_lang') ?? FALSE,
      '#description' => $this->t('This will be used as the default lang for all pages.'),
      '#access' => $this->currentUser()->hasPermission('Administer HTML lang override settings'),
      '#states' => [
        'visible' => [
          ':input[name="override_html_lang"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          ':input[name="override_html_lang"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['route_path_override'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Override HTML lang attribute for specific routes'),
      '#description' => $this->t('Add path and language code in format given in example path_of_page|language_code. Make sure to add a valid language code and Each path should be on a new line and should start with a forward slash(/). <br>For example: <br> /about-us|hi <br> /contact-page|ja'),
      '#default_value' => $config->get('route_path_override') ?? '',
      '#weight' => 100,
      '#access' => $this->currentUser()->hasPermission('Override HTML lang attribute'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $routes_input = $form_state->getValue('route_path_override');
    $lines = preg_split('/\r\n|\r|\n/', $routes_input);
    $line_number = 1;

    foreach ($lines as $line) {
      $line = trim($line);
      if ($line === '') {
        $line_number++;
        continue;
      }
      // Validate that the line matches the expected format.
      if (!preg_match('#^\/[^\|]*\|[a-zA-Z_-]{2,20}$#', $line)) {
        $form_state->setErrorByName('route_path_override', $this->t(
          'Line @line is invalid. Use the format: /page-path|language_code (e.g., /about-us|fr)',
          ['@line' => $line_number]
        ));
        break;
      }
      $line_number++;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('html_lang_override.settings')
      ->set('override_html_lang', $form_state->getValue('override_html_lang'))
      ->set('default_lang', $form_state->getValue('default_lang'))
      ->set('route_path_override', trim($form_state->getValue('route_path_override')))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
