<?php

namespace Drupal\Tests\html_tag_usage\Functional;

use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\Node;
use Drupal\taxonomy\Entity\Term;
use Drupal\Tests\BrowserTestBase;

/**
 * Provides functional tests for HTML Tag Usage module.
 *
 * @group html_tag_usage
 */
class HtmlTagUsageTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'filter',
    'node',
    'taxonomy',
    'html_tag_usage',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Users created during set-up.
   *
   * @var \Drupal\user\Entity\User[]
   */
  protected $users;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create users.
    $this->users['admin_user'] = $this->drupalCreateUser([
      'administer html tag usage',
      'access administration pages',
      'view html tag usage report',
      'generate html tag usage report',
    ]);
    $this->users['admin_site_config'] = $this->drupalCreateUser([
      'administer site configuration',
    ]);
    $this->users['access_content'] = $this->drupalCreateUser([
      'access content',
    ]);
    $this->users['view_report'] = $this->drupalCreateUser([
      'view html tag usage report',
    ]);
    $this->users['generate_report'] = $this->drupalCreateUser([
      'view html tag usage report',
      'generate html tag usage report',
    ]);

    // Create a filter format.
    \Drupal::entityTypeManager()->getStorage('filter_format')
      ->create([
        'format' => 'filtered_html',
        'name' => 'Filtered HTML',
      ])
      ->save();

    // Create a node type with a body field.
    $type = \Drupal::entityTypeManager()->getStorage('node_type')
      ->create([
        'type' => 'page',
        'name' => 'Page',
      ]);
    $type->save();
    $field_storage = FieldStorageConfig::loadByName('node', 'body');
    $field = \Drupal::entityTypeManager()
      ->getStorage('field_config')
      ->create([
        'field_storage' => $field_storage,
        'bundle' => $type->id(),
        'label' => 'Body',
        'settings' => [
          'display_summary' => TRUE,
          'allowed_formats' => [],
        ],
      ]);
    $field->save();

    // Create a vocabulary.
    \Drupal::entityTypeManager()->getStorage('taxonomy_vocabulary')
      ->create([
        'vid' => 'glossary',
        'name' => 'Glossary',
      ])
      ->save();
  }

  /**
   * Tests language switch links provided by Language Switcher Menu module.
   */
  public function testAnalyzer(): void {
    // Assert front page is available.
    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);

    // Check output for view report permission.
    $this->drupalLogin($this->users['view_report']);
    $this->drupalGet('/admin/config/development/html_tag_usage');
    $this->assertSession()->statusCodeEquals(403);
    $this->drupalGet('/admin/reports/html_tag_usage');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('The HTML tag usage report has never been generated.');
    $this->assertSession()->linkNotExists('Generate report');
    $this->assertSession()->linkNotExists('Regenerate report');

    // Check output for generate report permission.
    $this->drupalLogin($this->users['generate_report']);
    $this->drupalGet('/admin/config/development/html_tag_usage');
    $this->assertSession()->statusCodeEquals(403);
    $this->drupalGet('/admin/reports/html_tag_usage');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('The HTML tag usage report has never been generated.');
    $this->assertSession()->linkExists('Generate report');
    $this->assertSession()->linkNotExists('Regenerate report');

    // Check output for access content permission.
    $this->drupalLogin($this->users['access_content']);
    $this->drupalGet('/admin/config/development/html_tag_usage');
    $this->assertSession()->statusCodeEquals(403);
    $this->drupalGet('/admin/reports/html_tag_usage');
    $this->assertSession()->statusCodeEquals(403);

    // Check output for administer site configuration permission.
    $this->drupalLogin($this->users['admin_site_config']);
    $this->drupalGet('/admin/config/development/html_tag_usage');
    $this->assertSession()->statusCodeEquals(403);
    $this->drupalGet('/admin/reports/html_tag_usage');
    $this->assertSession()->statusCodeEquals(403);

    // Login as admin user.
    $this->drupalLogin($this->users['admin_user']);

    // Create a published node.
    $node = \Drupal::entityTypeManager()->getStorage('node')
      ->create([
        'type' => 'page',
        'title' => $this->randomMachineName(),
        'status' => 1,
        'body' => [
          0 => [
            'value' => '<p>In rare cases, the doctor may refuse to allow access to the patient file, either in part or in full. This is the case if access conflicts with therapeutic reasons or other rights of third parties. However, the refusal must be justified accordingly.</p>
<p></p><h2 class="content-tab">Problems</h2>
<p><strong>Problems</strong></p>
<p>Sometimes doctors refuse to show or hand over the files to patients. In this case, it is best to explicitly remind the doctor that you wish to exercise your legal right to access the files. If this does not help, the responsible <a href="http://www.bundesaerztekammer.de/page.asp?his=0.8.5585" target="_blank">medical association</a>&nbsp; or the <a href="http://www.bfdi.bund.de/cln_111/DE/AnschriftenUndLinks/Landesdatenschutzbeauftragte/AnschriftenLandesdatenschutzbeauftragte.html?nn=408930" target="_blank">state data protection officer</a>&nbsp; can help. Only if they are unable to help should you take legal action to obtain the file.</p>',
            'format' => 'filtered_html',
          ],
        ],
      ]);
    $node->save();
    self::assertInstanceOf(Node::class, $node);

    // Create a published term.
    $term = \Drupal::entityTypeManager()->getStorage('taxonomy_term')
      ->create([
        'vid' => 'glossary',
        'name' => $this->randomMachineName(),
        'status' => 1,
        'description' => [
          0 => [
            'value' => '<p>In rare cases, the doctor may refuse to allow access to the patient file, either in part or in full. This is the case if access conflicts with therapeutic reasons or other rights of third parties. However, the refusal must be justified accordingly.</p>
<p></p><h2 class="content-tab">Problems</h2>
<p><strong>Problems</strong></p>
<p>Sometimes doctors refuse to show or hand over the files to patients. In this case, it is best to explicitly remind the doctor that you wish to exercise your legal right to access the files. If this does not help, the responsible <a href="http://www.bundesaerztekammer.de/page.asp?his=0.8.5585" target="_blank">medical association</a>&nbsp; or the <a href="http://www.bfdi.bund.de/cln_111/DE/AnschriftenUndLinks/Landesdatenschutzbeauftragte/AnschriftenLandesdatenschutzbeauftragte.html?nn=408930" target="_blank">state data protection officer</a>&nbsp; can help. Only if they are unable to help should you take legal action to obtain the file.</p>',
            'format' => 'filtered_html',
          ],
        ],
      ]);
    $term->save();
    self::assertInstanceOf(Term::class, $term);

    // Assert config page is available.
    $this->drupalGet('/admin/config/development/html_tag_usage');
    $this->assertSession()->statusCodeEquals(200);

    // Assert correct output for initial view of report page.
    $this->drupalGet('/admin/reports/html_tag_usage');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('The HTML tag usage report has never been generated.');
    $this->assertSession()->pageTextNotContains('This HTML tag usage report has been generated at');
    $this->assertSession()->pageTextNotContains('Tags listing * as attribute value don\'t have any HTML attributes.');
    $this->assertSession()->linkExists('Generate report');
    $this->assertSession()->linkNotExists('Regenerate report');

    // Generate the report.
    $this->clickLink('Generate report');
    $this->checkForMetaRefresh();
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('This HTML tag usage report has been generated at');
    $this->assertSession()->pageTextContains('Tags listing * as attribute value don\'t have any HTML attributes.');
    $date = $this->assertSession()->elementExists('xpath', '//p[contains(.,"This HTML tag usage report has been generated at")]/em[@class="placeholder"]');
    self::assertNotNull($date);
    self::assertNotEmpty($date->getText());
    $this->assertSession()->linkNotExists('Generate report');
    $this->assertSession()->linkExists('Regenerate report');

    // Assert expected output for report table.
    $expected_table = [
      1 => [
        'filter_format' => 'Filtered HTML',
        'tag' => 'a',
        'attribute' => 'href',
        'count' => '4',
        'count_link_index' => 0,
        'entities' => [
          1 => [
            'entity' => $node,
            'field' => 'Body',
            'count' => '2',
          ],
          2 => [
            'entity' => $term,
            'field' => 'Description',
            'count' => '2',
          ],
        ],
      ],
      2 => [
        'filter_format' => 'Filtered HTML',
        'tag' => 'a',
        'attribute' => 'target',
        'count' => '4',
        'count_link_index' => 1,
        'entities' => [
          1 => [
            'entity' => $node,
            'field' => 'Body',
            'count' => '2',
          ],
          2 => [
            'entity' => $term,
            'field' => 'Description',
            'count' => '2',
          ],
        ],
      ],
      3 => [
        'filter_format' => 'Filtered HTML',
        'tag' => 'h2',
        'attribute' => 'class',
        'count' => '2',
        'count_link_index' => 0,
        'entities' => [
          1 => [
            'entity' => $node,
            'field' => 'Body',
            'count' => '1',
          ],
          2 => [
            'entity' => $term,
            'field' => 'Description',
            'count' => '1',
          ],
        ],
      ],
      4 => [
        'filter_format' => 'Filtered HTML',
        'tag' => 'p',
        'attribute' => '*',
        'count' => '8',
        'count_link_index' => 0,
        'entities' => [
          1 => [
            'entity' => $node,
            'field' => 'Body',
            'count' => '4',
          ],
          2 => [
            'entity' => $term,
            'field' => 'Description',
            'count' => '4',
          ],
        ],
      ],
      5 => [
        'filter_format' => 'Filtered HTML',
        'tag' => 'strong',
        'attribute' => '*',
        'count' => '2',
        'count_link_index' => 1,
        'entities' => [
          1 => [
            'entity' => $node,
            'field' => 'Body',
            'count' => '1',
          ],
          2 => [
            'entity' => $term,
            'field' => 'Description',
            'count' => '1',
          ],
        ],
      ],
    ];
    $this->assertTable($expected_table);

    // Create a node with an invalid text format.
    $node_invalid = \Drupal::entityTypeManager()->getStorage('node')
      ->create([
        'type' => 'page',
        'title' => $this->randomMachineName(),
        'status' => 1,
        'body' => [
          0 => [
            'value' => '<p>In rare cases, the doctor may refuse to allow access to the patient file, either in part or in full. This is the case if access conflicts with therapeutic reasons or other rights of third parties. However, the refusal must be justified accordingly.</p>',
            'format' => 'invalid_format',
          ],
        ],
      ]);
    $node_invalid->save();
    self::assertInstanceOf(Node::class, $node_invalid);

    // Regenerate report.
    $this->regenerateReport();

    // Assert expected output of report table.
    $expected_table[6] = [
      'filter_format' => 'Invalid text format id invalid_format',
      'tag' => 'p',
      'attribute' => '*',
      'count' => '1',
      'count_link_index' => 0,
      'entities' => [
        1 => [
          'entity' => $node_invalid,
          'field' => 'Body',
          'count' => '1',
        ],
      ],
    ];
    $this->assertTable($expected_table);

    // Create a node with colons and underscores in tags and attributes.
    $node_colon_underscore = \Drupal::entityTypeManager()->getStorage('node')
      ->create([
        'type' => 'page',
        'title' => $this->randomMachineName(),
        'status' => 1,
        'body' => [
          0 => [
            'value' => '<p><span test_attribute="underscore" style="text-align: justify;">Patients have the right to inspect their patient files at any time, provided that there are no significant therapeutic reasons or other significant rights of third parties that would prevent this.&nbsp;</span><span style="text-align: justify;">This includes all documents that are necessary from a professional point of view for current and future treatment, in particular medical history, diagnosis, examinations (results), findings, therapies and their effects, interventions, as well as consents, explanations, and doctor\'s letters.</span><span style="text-align: justify;">&nbsp;In the case of electronic files, the physician must provide the patient with copies of the patient files, but may charge for the cost of copying.</span></p>
<p>X-rays and other documents that cannot be easily copied must be handed over to the patient by the doctor. This can be important, for example, if you want to consult another doctor for a second opinion. However, the doctor may require the patient to return the documents afterwards.<o:p></o:p></p>
<box_no></box_no><h2 class="content-tab">Limitations</h2>
<p><span lang="en" xml:lang="en">Yearly report of European Centre for Disease Prevention and Control</span></p>',
            'format' => 'underscore_colon_format',
          ],
        ],
      ]);
    $node_colon_underscore->save();
    self::assertInstanceOf(Node::class, $node_colon_underscore);

    // Regenerate report.
    $this->regenerateReport();

    // Assert expected output of report table.
    $expected_table[7] = [
      'filter_format' => 'Invalid text format id underscore_colon_format',
      'tag' => 'box_no',
      'attribute' => '*',
      'count' => '1',
      'count_link_index' => 1,
      'entities' => [
        1 => [
          'entity' => $node_colon_underscore,
          'field' => 'Body',
          'count' => '1',
        ],
      ],
    ];
    $expected_table[8] = [
      'filter_format' => 'Invalid text format id underscore_colon_format',
      'tag' => 'h2',
      'attribute' => 'class',
      'count' => '1',
      'count_link_index' => 2,
      'entities' => [
        1 => [
          'entity' => $node_colon_underscore,
          'field' => 'Body',
          'count' => '1',
        ],
      ],
    ];
    $expected_table[9] = [
      'filter_format' => 'Invalid text format id underscore_colon_format',
      'tag' => 'o:p',
      'attribute' => '*',
      'count' => '1',
      'count_link_index' => 3,
      'entities' => [
        1 => [
          'entity' => $node_colon_underscore,
          'field' => 'Body',
          'count' => '1',
        ],
      ],
    ];
    $expected_table[10] = [
      'filter_format' => 'Invalid text format id underscore_colon_format',
      'tag' => 'p',
      'attribute' => '*',
      'count' => '3',
      'count_link_index' => 0,
      'entities' => [
        1 => [
          'entity' => $node_colon_underscore,
          'field' => 'Body',
          'count' => '3',
        ],
      ],
    ];
    $expected_table[11] = [
      'filter_format' => 'Invalid text format id underscore_colon_format',
      'tag' => 'span',
      'attribute' => 'lang',
      'count' => '1',
      'count_link_index' => 4,
      'entities' => [
        1 => [
          'entity' => $node_colon_underscore,
          'field' => 'Body',
          'count' => '1',
        ],
      ],
    ];
    $expected_table[12] = [
      'filter_format' => 'Invalid text format id underscore_colon_format',
      'tag' => 'span',
      'attribute' => 'style',
      'count' => '3',
      'count_link_index' => 1,
      'entities' => [
        1 => [
          'entity' => $node_colon_underscore,
          'field' => 'Body',
          'count' => '3',
        ],
      ],
    ];
    $expected_table[13] = [
      'filter_format' => 'Invalid text format id underscore_colon_format',
      'tag' => 'span',
      'attribute' => 'test_attribute',
      'count' => '1',
      'count_link_index' => 5,
      'entities' => [
        1 => [
          'entity' => $node_colon_underscore,
          'field' => 'Body',
          'count' => '1',
        ],
      ],
    ];
    $expected_table[14] = [
      'filter_format' => 'Invalid text format id underscore_colon_format',
      'tag' => 'span',
      'attribute' => 'xml:lang',
      'count' => '1',
      'count_link_index' => 6,
      'entities' => [
        1 => [
          'entity' => $node_colon_underscore,
          'field' => 'Body',
          'count' => '1',
        ],
      ],
    ];
    $this->assertTable($expected_table);

    // Assert HTML filter configuration on report page.
    $this->assertFilterConfig([
      'Filtered HTML' => '<a href target> <h2 class> <p> <strong>',
      'Invalid text format id invalid_format' => '<p>',
    ]);

    // Reconfigure the module keeping default settings.
    $this->reconfigure();
  }

  /**
   * Reconfigure HTML Tag Usage module.
   *
   * @param array<string, mixed> $edit
   *   Field data in an associative array, if any. Changes the current input
   *   fields (where possible) to the values indicated. A checkbox can be set to
   *   TRUE to be checked and should be set to FALSE to be unchecked.
   */
  protected function reconfigure(array $edit = []): void {
    $this->drupalGet('/admin/config/development/html_tag_usage');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('The configuration options have been saved.');
  }

  /**
   * Regenerates the global HTML tag usage report.
   */
  protected function regenerateReport(): void {
    $this->drupalGet('/admin/reports/html_tag_usage');
    $this->assertSession()->statusCodeEquals(200);
    $this->clickLink('Regenerate report');
    $this->checkForMetaRefresh();
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Asserts expected output for report table.
   *
   * @param array<int,array<string,mixed>> $expected_table
   *   Expected data for output.
   *
   * @phpstan-param array<int, array{
   *   'filter_format': string,
   *   'tag': string,
   *   'attribute': string,
   *   'count': string,
   *   'count_link_index': int,
   *   'entities': array<int, array{
   *     'entity': \Drupal\Core\Entity\ContentEntityInterface,
   *     'field': string,
   *     'count': string,
   *   }>,
   * }> $expected_table
   */
  protected function assertTable(array $expected_table): void {
    foreach ($expected_table as $row => $expected) {
      $this->assertTableRow($row, $expected);
    }
  }

  /**
   * Asserts expected output for report table row.
   *
   * @param int $row
   *   Row to assert expected output for starting at 1.
   * @param array<string,mixed> $expected
   *   Expected data for output.
   *
   * @phpstan-param array{
   *   'filter_format': string,
   *   'tag': string,
   *   'attribute': string,
   *   'count': string,
   *   'count_link_index': int,
   *   'entities': array<int, array{
   *     'entity': \Drupal\Core\Entity\ContentEntityInterface,
   *     'field': string,
   *     'count': string,
   *   }>,
   * } $expected
   */
  protected function assertTableRow(int $row, array $expected): void {
    $this->drupalGet('/admin/reports/html_tag_usage');
    $this->assertSession()->statusCodeEquals(200);

    $table = $this->assertSession()->elementExists('xpath', '//table');
    self::assertNotNull($table);
    $caption = $table->find('xpath', '//caption');
    self::assertSame($caption->getText(), 'Tags with attributes by text format');

    $format = $this->assertSession()->elementExists('xpath', '//table/tbody/tr[' . $row . ']/td[1]');
    self::assertSame($format->getText(), $expected['filter_format']);
    $tag = $this->assertSession()->elementExists('xpath', '//table/tbody/tr[' . $row . ']/td[2]');
    self::assertSame($tag->getText(), $expected['tag']);
    $attribute = $this->assertSession()->elementExists('xpath', '//table/tbody/tr[' . $row . ']/td[3]');
    self::assertSame($attribute->getText(), $expected['attribute']);
    $count = $this->assertSession()->elementExists('xpath', '//table/tbody/tr[' . $row . ']/td[4]');
    $count_link = $count->find('xpath', '//a');
    self::assertSame($count_link->getText(), $expected['count']);
    $this->clickLink($count_link->getText(), $expected['count_link_index']);
    $this->assertSession()->statusCodeEquals(200);

    $table = $this->assertSession()->elementExists('xpath', '//table');
    self::assertNotNull($table);
    $this->assertSession()->pageTextContains('Inspect ' . $expected['tag'] . '[' . $expected['attribute'] . '] for ' . $expected['filter_format']);

    foreach ($expected['entities'] as $entity_row => $entity_expected) {
      $this->assertEntityTableRow($entity_row, $entity_expected);
    }
  }

  /**
   * Asserts expected output for entity table row.
   *
   * @param int $row
   *   Row to assert expected output for starting at 1.
   * @param array<string,mixed> $expected
   *   Expected data for output. The expected entity, the field label and count.
   *
   * @phpstan-param array{
   *   'entity': \Drupal\Core\Entity\ContentEntityInterface,
   *   'field': string,
   *   'count': string,
   * } $expected
   */
  protected function assertEntityTableRow(int $row, array $expected): void {
    $entity_type = $this->assertSession()->elementExists('xpath', '//table/tbody/tr[' . $row . ']/td[1]');
    self::assertSame($entity_type->getText(), (string) $expected['entity']->getEntityType()->getSingularLabel());
    $id = $this->assertSession()->elementExists('xpath', '//table/tbody/tr[' . $row . ']/td[2]');
    self::assertSame($id->getText(), $expected['entity']->id());
    $language = $this->assertSession()->elementExists('xpath', '//table/tbody/tr[' . $row . ']/td[3]');
    self::assertSame($language->getText(), $expected['entity']->language()->getName());
    $field = $this->assertSession()->elementExists('xpath', '//table/tbody/tr[' . $row . ']/td[4]');
    self::assertSame($field->getText(), $expected['field']);
    $entity_link = $this->assertSession()->elementExists('xpath', '//table/tbody/tr[' . $row . ']/td[5]/a');
    self::assertSame($entity_link->getText(), $expected['entity']->label());
    self::assertSame($entity_link->getAttribute('href'), $expected['entity']->toUrl('edit-form')->toString());
    $count = $this->assertSession()->elementExists('xpath', '//table/tbody/tr[' . $row . ']/td[6]');
    self::assertSame($count->getText(), $expected['count']);
  }

  /**
   * Asserts expected filter configuration.
   *
   * @param array<string,string> $expected
   *   The expected label (key) and configuration (value).
   */
  protected function assertFilterConfig(array $expected): void {
    $this->drupalGet('/admin/reports/html_tag_usage');
    $this->assertSession()->statusCodeEquals(200);

    $this->assertSession()->pageTextContains('HTML Filter Configuration');
    $this->assertSession()->pageTextContains('Use the following HTML filter configuration to render all HTML elements currently in use. Note: It may be insecure to use the generated configuration as is. You are encouraged to review it before use.');
    foreach ($expected as $label => $config) {
      // @todo Assert that the config actually belongs to the label using xpath.
      $this->assertSession()->pageTextContains($label);
      $this->assertSession()->pageTextContains($config);
    }
  }

}
