<?php

declare(strict_types=1);

namespace Drupal\htmx\Plugin\views\pager;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsPager;
use Drupal\views\Plugin\views\pager\Mini;

/**
 * The plugin to handle mini pager using HTMX.
 *
 * @ingroup views_pager_plugins
 */
#[ViewsPager(
  id: "htmx_mini",
  title: new TranslatableMarkup("HTMX mini pager"),
  short_title: new TranslatableMarkup("HTMX"),
  help: new TranslatableMarkup("A simple pager containing previous and next links that load using HTMX."),
  theme: "htmx_mini_pager",
)]
class HtmxMini extends Mini {

  /**
   * {@inheritdoc}
   */
  public function render($input) {
    $build = parent::render($input);
    $build['#view'] = $this->view;
    $build['#attached']['library'][] = 'core/drupal.htmx';
    $build['#attached']['library'][] = 'htmx/views';
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function defineOptions() {
    $options = parent::defineOptions();
    $options['htmx_options'] = [
      'contains' => [
        'pager_push_url' => ['default' => FALSE],
      ],
    ];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state):void {
    parent::buildOptionsForm($form, $form_state);

    $form['htmx_options'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#tree' => TRUE,
      '#title' => $this->t('HTMX Options'),
      '#input' => TRUE,
    ];

    $form['htmx_options']['pager_push_url'] = [
      '#description' => $this->t('The Urls will be updated in the address bar'),
      '#type' => 'checkbox',
      '#weight' => 1,
      '#title' => $this->t('Push Url'),
      '#default_value' => $this->options['htmx_options']['pager_push_url'] ? 1 : 0,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state):void {
    parent::submitOptionsForm($form, $form_state);
    $this->options['htmx_options']['pager_push_url'] = !empty($form_state->getValue('pager_options')['htmx_options']['pager_push_url']) ? 1 : 0;
  }

}
