<?php

declare(strict_types=1);

namespace Drupal\htmx\Entity;

use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Htmx\Htmx;
use Drupal\Core\Template\Attribute;
use Drupal\Core\Template\AttributeHelper;
use Drupal\Core\Url;

/**
 * Provides a listing of htmx blocks.
 */
final class HtmxBlockListBuilder extends ConfigEntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['label'] = $this->t('Label');
    $header['id'] = $this->t('Machine name');
    $header['status'] = $this->t('Status');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\htmx\Entity\HtmxBlockInterface $entity */
    $row['label'] = $entity->label();
    $row['id'] = $entity->id();
    $row['status'] = $entity->status() ? $this->t('Enabled') : $this->t('Disabled');
    return $row + parent::buildRow($entity);
  }

  /**
   * Build the list render array.
   *
   * @return mixed[]
   *   The render array.
   */
  public function render(): array {
    // Create an HTMX powered button.
    $addAttributes = new Attribute([
      'class' => [
        'button',
        'button--action',
        'button--primary',
        'button--htmx',
        'button--htmx-refresh',
      ],
      'aria-live' => 'polite',
      'name' => 'open-plugin-dialog',
    ]);
    $htmxAdd = new Htmx();
    $htmxAdd->get(Url::fromRoute('htmx_blocks.htmx_system_block_library'))
      ->select('#drupal-off-canvas-wrapper')
      ->target('main')
      ->swap('afterbegin')
      ->on('::afterOnLoad', 'setListWithDialogOpen(this)')
      ->on('pluginDialogRemoved', 'setListWithoutDialog(this)');
    $addAttributes->merge($htmxAdd->getAttributes());
    $button['button-add'] = [
      '#type' => 'inline_template',
      '#template' => '<button {{ attributes }}><span class="default-text">{{ add }}</span> <span class="htmx-indicator">{{ load }}</span></button>',
      '#context' => [
        'add' => $this->t('Add a block'),
        'load' => 'Loading...',
        'attributes' => $addAttributes,
      ],
    ];

    // Build the list.
    $table = parent::render();
    // Assemble the parts.
    // @todo recreate the filter JS without JQuery.
    $build = [
      '#type' => 'container',
      '#attributes' => [
        'class' => [
          'htmx-block-list-container',
        ],
      ],
      'button' => $button,
      'list-container' => [
        '#type' => 'container',
        '#attributes' => [
          'id' => 'htmx-block-list',
        ],
        'list' => $table,
      ],
      '#attached' => [
        'library' => [
          'core/drupal.dropbutton',
          'core/drupal.htmx',
          'htmx/dialog.off-canvas',
          'block/drupal.block.admin',
        ],
      ],
    ];
    return $build;
  }

  /**
   * Build a render array for the operations of a single entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   Operations will be built for this entity.
   *
   * @return mixed[]
   *   The render array.
   */
  public function buildOperations(EntityInterface $entity) {
    $htmx = new Htmx();
    $htmx->boost(TRUE);
    $build = [
      '#type' => 'operations',
      '#links' => $this->getOperations($entity),
    ];
    $htmx->applyTo($build);
    return $build;
  }

  /**
   * Build the operations render array with HTMX properties.
   *
   * @return mixed[]
   *   The render array.
   */
  public function getDefaultOperations(EntityInterface $entity): array {
    $operations = parent::getDefaultOperations($entity);
    $htmx = new Htmx();
    $htmx->select('#drupal-off-canvas-wrapper')
      ->target('main')
      ->swap('afterbegin')
      ->pushUrl(FALSE);
    foreach ($operations as &$operation) {
      if (isset($operation['url']) && $operation['url'] instanceof Url) {
        $attributes = $operation['url']->getOption('attributes') ?: [];
        $attributes = AttributeHelper::mergeCollections($attributes, $htmx->getAttributes());
        $operation['url']->setOption('attributes', $attributes);
      }
    }
    if ($entity->access('update')) {
      $editUrl = Url::fromRoute('entity.htmx_block.edit_form', ['htmx_block' => $entity->id()]);
      $attributes = $editUrl->getOption('attributes') ?: [];
      $attributes = AttributeHelper::mergeCollections($attributes, $htmx->getAttributes());
      $editUrl->setOption('attributes', $attributes);
      $operations['edit'] = [
        'title' => $this->t('Edit'),
        'weight' => 10,
        'url' => $editUrl,
      ];
    }
    if ($entity->access('delete')) {
      $deleteUrl = Url::fromRoute('entity.htmx_block.delete_form', ['htmx_block' => $entity->id()]);
      $attributes = $deleteUrl->getOption('attributes') ?: [];
      $attributes = AttributeHelper::mergeCollections($attributes, $htmx->getAttributes());
      $deleteUrl->setOption('attributes', $attributes);
      $operations['delete'] = [
        'title' => $this->t('Delete'),
        'weight' => 100,
        'url' => $deleteUrl,
      ];
    }
    return $operations;
  }

}
