<?php

namespace Drupal\htmx_extras\Controller;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Cache\CacheableResponse;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class LazyLoadController extends ControllerBase {

  private ?EntityInterface $entity = NULL;

  public function __construct(
    private readonly RendererInterface $renderer,
  ) {}

  /**
   * Render the entity in the requested view mode.
   */
  public function entity(string $entity_type, string $entity_id, string $view_mode, ?string $revision_id): Response {
    $entity = $this->getEntity($entity_type, $entity_id, $revision_id);
    $view_builder = $this->entityTypeManager()->getViewBuilder($entity->getEntityTypeId());
    $build = $view_builder->view($entity, $view_mode);
    $response = new CacheableResponse($this->renderer->renderRoot($build));
    $response->getCacheableMetadata()
      ->addCacheContexts([
        'url.query_args:view_mode_name',
      ])
      ->addCacheableDependency($entity);

    return $response;
  }

  /**
   * Check access for the entity.
   */
  public  function entityAccess(AccountInterface $account, string $entity_type, string $entity_id, string $view_mode, ?string $revision_id): AccessResultInterface {
    if (!$entity = $this->getEntity($entity_type, $entity_id, $revision_id)) {
      throw new NotFoundHttpException();
    }
    return $entity->access('view', $account, TRUE);
  }

  /**
   * Retrieve and static cache an entity based on the given parameters.
   */
  private function getEntity(string $entity_type, string|int $entity_id, string|int|null $revision_id): ?EntityInterface {
    if (!isset($this->entity)) {
      $storage = $this->entityTypeManager()->getStorage($entity_type);
      if (!empty($revision_id) && method_exists($storage, 'loadRevision')) {
        $this->entity = $storage->loadRevision($revision_id);
      }
      else {
        $this->entity = $storage->load($entity_id);
      }
    }

    return $this->entity;
  }

}
