<?php

namespace Drupal\htmx_extras\Render;

use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\RefinableCacheableDependencyInterface;
use Drupal\Core\Cache\RefinableCacheableDependencyTrait;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\htmx_extras\HtmxExtrasHelper;

final class HtmxEntityPartial implements RefinableCacheableDependencyInterface {

  use RefinableCacheableDependencyTrait;

  public function __construct(
    private readonly string $entityType,
    private readonly string|int $entityId,
    private readonly ?string $bundle = NULL,
    private ?string $viewMode = NULL,
    private readonly ?int $revisionId = NULL,
    private array $options = [],
    private array $extraHtmxAttributes = [
      'data-hx-trigger' => 'revealed',
      'data-hx-swap' => 'outerHTML',
    ],
    private array $attachments = [],
    ?CacheableDependencyInterface $cache = NULL,
  ) {
    if ($cache) {
      $this->setCacheability($cache);
    }
  }

  public static function fromEntity(EntityInterface $entity): self {
    return new self(
      entityType: $entity->getEntityTypeId(),
      entityId: $entity->id(),
      bundle: $entity->bundle(),
      revisionId: $entity instanceof RevisionableInterface ? $entity->getRevisionId() : NULL,
      options: ['language' => $entity->language()],
      cache: (new CacheableMetadata())->addCacheableDependency($entity),
    );
  }

  public static function fromParameters(
    string $entity_type,
    string|int $entity_id,
    ?string $bundle = NULL,
    ?int $revision_id = NULL,
    ?LanguageInterface $language = NULL,
  ): self {
    $options = [];
    if ($language) {
      $options['language'] = $language;
    }
    return new self(
      entityType: $entity_type,
      entityId: $entity_id,
      bundle: $bundle,
      revisionId: $revision_id,
      options: $options,
      cache: (new CacheableMetadata())
        ->addCacheTags(["{$entity_type}_list", "$entity_type:$entity_id"])
        ->addCacheContexts(['languages'])
        ->setCacheMaxAge(-1)
        ->addCacheableDependency($language),
    );
  }

  /**
   * Set a HTMX parameter (excluding the data-hx or hx- prefixes).
   *
   * @see \Drupal\htmx_extras\HtmxExtrasHelper::availableAttributes()
   *   For the available attributes/directives.
   *
   * @throws \BadMethodCallException
   */
  public function setHtmxParameter(string $key, string $value): self {
    $key = str_replace(['data-hx-', 'hx-'], '', $key);
    if (!in_array($key, HtmxExtrasHelper::availableAttributes(with_prefix: FALSE))) {
      throw new \BadMethodCallException('This HTMX directive does not exist.');
    }

    $this->extraHtmxAttributes[$key] = $value;
    return $this;
  }

  /**
   * Set a route/url option, note that the language is usually already set.
   */
  public function setOption(string $key, mixed $value): self {
    $this->options[$key] = $value;
    return $this;
  }

  /**
   * Sets the entity view mode.
   */
  public function setViewMode(string $value): self {
    $this->viewMode = $value;
    return $this;
  }

  /**
   * Add an attachment to a given subset.
   */
  public function addAttachment(string $key, mixed $value): self {
    $this->attachments[$key][] = $value;
    return $this;
  }

  /**
   * Override an entire attachment key.
   */
  public function setAttachment(string $key, mixed $value): self {
    $this->attachments[$key] = $value;
    return $this;
  }

  /**
   * Returns the final template render array.
   */
  public function render(): array {
    $extra_htmx_attributes = [];
    foreach ($this->extraHtmxAttributes as $key => $value) {
      if (str_starts_with($key, 'data-hx-')) {
        $extra_htmx_attributes[$key] = $value;
      }
      elseif (str_starts_with($key, 'hx-')) {
        $extra_htmx_attributes["data-$key"] = $value;
      }
      else {
        $extra_htmx_attributes["data-hx-$key"] = $value;
      }
    }
    return [
      '#theme' => 'htmx_entity_partial',
      '#entity_type' => $this->entityType,
      '#entity_id' => $this->entityId,
      '#revision_id' => $this->revisionId,
      '#bundle' => $this->bundle,
      '#view_mode' => $this->viewMode,
      '#extra_htmx_attributes' => $extra_htmx_attributes,
      '#options' => $this->options,
      '#cache' => [
        'tags' => $this->getCacheTags(),
        'contexts' => $this->getCacheContexts(),
        'max-age' => $this->getCacheMaxAge(),
      ],
      '#attached' => $this->attachments,
    ];
  }

}
