<?php

namespace Drupal\htmx_extras\Plugin\Derivative;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides block plugin definitions for all Views block displays.
 */
class HtmxViewBlock implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * List of derivative definitions.
   *
   * @var array
   */
  protected $derivatives = [];

  /**
   * The base plugin ID.
   *
   * @var string
   */
  protected $basePluginId;

  /**
   * The htmx view storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $htmxViewStorage;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $base_plugin_id,
      $container->get('entity_type.manager')->getStorage('htmx_view'),
    );
  }

  /**
   * Constructs a ViewsBlock object.
   *
   * @param string $base_plugin_id
   *   The base plugin ID.
   * @param \Drupal\Core\Entity\EntityStorageInterface $view_storage
   *   The entity storage to load views.
   */
  public function __construct($base_plugin_id, EntityStorageInterface $view_storage) {
    $this->basePluginId = $base_plugin_id;
    $this->htmxViewStorage = $view_storage;
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinition($derivative_id, $base_plugin_definition) {
    if (!empty($this->derivatives) && !empty($this->derivatives[$derivative_id])) {
      return $this->derivatives[$derivative_id];
    }
    $this->getDerivativeDefinitions($base_plugin_definition);
    return $this->derivatives[$derivative_id];
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    // Check all Views for block displays.
    /** @var \Drupal\htmx_extras\HtmxViewInterface $htmx_view */
    foreach ($this->htmxViewStorage->loadMultiple() as $htmx_view) {
      // Do not return results for disabled views.
      if (!$htmx_view->status() || $htmx_view->getViewType() !== 'block') {
        continue;
      }

      $delta = $htmx_view->id();
      $this->derivatives[$delta] = [
        'category' => $this->t('HTMX View'),
        'admin_label' => $htmx_view->label(),
        'config_dependencies' => [
          'config' => [
            $htmx_view->getConfigDependencyName(),
          ],
        ],
      ];

      $this->derivatives[$delta] += $base_plugin_definition;
    }
    return $this->derivatives;
  }

}
