<?php

namespace Drupal\htmx_extras\Controller;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Render\HtmlResponse;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\htmx_extras\HtmxViewInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class LazyLoadController extends ControllerBase {

  private ?EntityInterface $entity = NULL;

  public function __construct(
    private readonly RendererInterface $renderer,
  ) {}

  /**
   * Render the entity in the requested view mode.
   */
  public function entity(string $entity_type, string $entity_id, string $view_mode, ?string $revision_id): HtmlResponse {
    $entity = $this->getEntity($entity_type, $entity_id, $revision_id);
    $view_builder = $this->entityTypeManager()->getViewBuilder($entity->getEntityTypeId());
    $build = $view_builder->view($entity, $view_mode);
    $response = $this->createResponseForBuild($build);
    $response->getCacheableMetadata()
      ->addCacheContexts([
        'url.query_args:view_mode_name',
      ])
      ->addCacheableDependency($entity);

    return $response;
  }

  /**
   * Check access for the entity.
   */
  public  function entityAccess(AccountInterface $account, string $entity_type, string $entity_id, string $view_mode, ?string $revision_id): AccessResultInterface {
    if (!$entity = $this->getEntity($entity_type, $entity_id, $revision_id)) {
      throw new NotFoundHttpException();
    }
    return $entity->access('view', $account, TRUE);
  }

  public function view(HtmxViewInterface $htmx_view): array {
    return $htmx_view->render();
  }

  public function viewPlain(HtmxViewInterface $htmx_view): HtmlResponse {
    $response = $this->createResponseForBuild($this->view($htmx_view));
    $response->addCacheableDependency($htmx_view);
    return $response;
  }

  /**
   * Retrieve and static cache an entity based on the given parameters.
   */
  private function getEntity(string $entity_type, string|int $entity_id, string|int|null $revision_id, ?string $langcode = NULL): ?EntityInterface {
    if (!isset($this->entity)) {
      $langcode = $langcode ?: $this->languageManager()->getCurrentLanguage()->getId();
      $storage = $this->entityTypeManager()->getStorage($entity_type);
      if (!empty($revision_id) && method_exists($storage, 'loadRevision')) {
        $this->entity = $storage->loadRevision($revision_id);
      }
      else {
        $this->entity = $storage->load($entity_id);
      }

      if ($this->entity instanceof TranslatableInterface && $this->entity->hasTranslation($langcode)) {
        $this->entity = $this->entity->getTranslation($langcode);
      }
    }

    return $this->entity;
  }

  private function createResponseForBuild(array $build): HtmlResponse {
    $types = [
      'styles' => 'css',
      'scripts' => 'js',
      'scripts_bottom' => 'js-bottom',
    ];
    $build['#placeholder_token'] = Crypt::randomBytesBase64(55);
    $placeholders = [];
    foreach ($types as $type => $placeholder_name) {
      $placeholder = '<' . $placeholder_name . '-placeholder token="' . $build['#placeholder_token'] . '">';
      $placeholders[$type] = $placeholder;
    }
    $build['#attached']['html_response_attachment_placeholders'] = $placeholders;

    $render = $placeholders['scripts']
      . $placeholders['styles']
      . $this->renderer->renderRoot($build)
      . $placeholders['scripts_bottom'];

    $response = new HtmlResponse($render);
    $response->addAttachments($build['#attached']);
    return $response;
  }

}
