<?php

namespace Drupal\htmx_extras\Controller;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Render\HtmlResponse;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\htmx_extras\HtmxViewInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class LazyLoadController extends ControllerBase {

  public function __construct(
    private readonly RendererInterface $renderer,
  ) {}

  /**
   * Render the entity in the requested view mode.
   */
  public function entity(string $entity_type, string $entity_id, string $view_mode, ?string $revision_id): HtmlResponse {
    $entity = $this->getEntity($entity_type, $entity_id, $revision_id);
    $view_builder = $this->entityTypeManager()->getViewBuilder($entity->getEntityTypeId());
    $view = $view_builder->view($entity, $view_mode);
    $build = [
      '#markup' => $this->renderer->renderRoot($view),
    ] + ($view['#attached'] ?? []);
    return new HtmlResponse($build);
  }

  /**
   * Check access for the entity.
   */
  public  function entityAccess(AccountInterface $account, string $entity_type, string $entity_id, string $view_mode, ?string $revision_id): AccessResultInterface {
    if (!$entity = $this->getEntity($entity_type, $entity_id, $revision_id)) {
      throw new NotFoundHttpException();
    }
    return $entity->access('view', $account, TRUE);
  }

  public function view(HtmxViewInterface $htmx_view): array {
    return $htmx_view->render();
  }

  public function viewPlain(HtmxViewInterface $htmx_view): HtmlResponse {
    $view = $htmx_view->render();
    $build = [
      '#markup' => $this->renderer->renderRoot($view),
    ] + ($view['#attached'] ?? []);
    return new HtmlResponse($build);
  }

  /**
   * Retrieve and static cache an entity based on the given parameters.
   */
  private function getEntity(string $entity_type, string|int $entity_id, string|int|null $revision_id, ?string $langcode = NULL): ?EntityInterface {
    $langcode = $langcode ?: $this->languageManager()->getCurrentLanguage()->getId();
    $storage = $this->entityTypeManager()->getStorage($entity_type);
    if (!empty($revision_id) && method_exists($storage, 'loadRevision')) {
      $entity = $storage->loadRevision($revision_id);
    }
    else {
      $entity = $storage->load($entity_id);
    }

    if ($entity instanceof TranslatableInterface && $entity->hasTranslation($langcode)) {
      $entity = $entity->getTranslation($langcode);
    }

    return $entity;
  }

}
