<?php

declare(strict_types=1);

namespace Drupal\htmx_extras\Entity;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\htmx_extras\Form\HtmxViewDuplicateForm;
use Drupal\htmx_extras\Render\HtmxEntityPartial;
use Drupal\htmx_extras\Form\HtmxViewForm;
use Drupal\htmx_extras\HtmxViewInterface;
use Drupal\htmx_extras\HtmxViewListBuilder;
use Drupal\views\ViewExecutable;
use Drupal\views\Views;

/**
 * Defines the htmx view entity type.
 */
#[ConfigEntityType(
  id: 'htmx_view',
  label: new TranslatableMarkup('HTMX View'),
  label_collection: new TranslatableMarkup('HTMX Views'),
  label_singular: new TranslatableMarkup('htmx view'),
  label_plural: new TranslatableMarkup('htmx views'),
  config_prefix: 'htmx_view',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'uuid' => 'uuid',
  ],
  handlers: [
    'list_builder' => HtmxViewListBuilder::class,
    'form' => [
      'add' => HtmxViewForm::class,
      'edit' => HtmxViewForm::class,
      'duplicate' => HtmxViewDuplicateForm::class,
      'delete' => EntityDeleteForm::class,
    ],
  ],
  links: [
    'collection' => '/admin/structure/htmx-view',
    'add-form' => '/admin/structure/htmx-view/add',
    'edit-form' => '/admin/structure/htmx-view/{htmx_view}',
    'duplicate-form' => '/admin/structure/htmx-view/{htmx_view}/duplicate',
    'delete-form' => '/admin/structure/htmx-view/{htmx_view}/delete',
  ],
  admin_permission: 'administer htmx_view',
  label_count: [
    'singular' => '@count htmx view',
    'plural' => '@count htmx views',
  ],
  config_export: [
    'id',
    'label',
    'description',
    'view_source',
    'mapped_keys',
    'entity_view_mode',
    'view_type',
    'title',
    'path',
    'pager',
    'facets',
    'show_facets',
    'show_facet_summary',
    'lazy_load_all',
    'lazy_load_all_settings',
  ],
)]
final class HtmxView extends ConfigEntityBase implements HtmxViewInterface {

  protected ?string $id;

  protected string $label;

  protected string $description;

  protected string $view_source;

  protected array $mapped_keys;

  protected string $entity_view_mode;

  protected string $view_type;

  protected string $title;

  protected string $path;

  protected array $pager = [];

  protected ?array $facets = [];

  protected ?bool $show_facets = TRUE;

  protected ?bool $show_facet_summary = FALSE;

  protected bool $lazy_load_all;

  protected ?array $lazy_load_all_settings = [];

  public function getViewSource(): ?ViewExecutable {
    if (!$view_source = $this->get('view_source')) {
      return NULL;
    }
    [$view_id, $display_id] = explode(':', $view_source, 2);
    $view = Views::getView($view_id);
    $view->setDisplay($display_id);
    return $view;
  }

  public function getMappedKey(string $key): ?string {
    return $this->getMappedKeys()[$key] ?? NULL;
  }

  public function getMappedKeys(): array {
    return $this->get('mapped_keys') ?: [];
  }

  public function getEntityViewMode(): ?string {
    return $this->get('entity_view_mode');
  }

  public function getViewType(): ?string {
    return $this->get('view_type');
  }

  public function getTitle(): ?string {
    return $this->get('title');
  }

  public function getPath(): ?string {
    return $this->get('path');
  }

  public function getPagerDefinition(): ?array {
    return $this->get('pager');
  }

  public function getFacetsConfiguration(): ?array {
    return $this->get('facets');
  }

  public function showFacets(): bool {
    return (bool) $this->get('show_facets');
  }

  public function showFacetSummary(): bool {
    return (bool) $this->get('show_facet_summary');
  }

  public function getLazyLoadAll(): bool {
    return (bool) $this->get('lazy_load_all');
  }

  public function getLazyLoadAllSettings(): array {
    return (array) $this->get('lazy_load_all_settings');
  }

  public function render(): array {
    $request = \Drupal::request();
    $view = $this->getViewSource();
    $view_id = $view->id();
    $display_id = $view->current_display;
    $entity_view_mode = $this->getentityViewMode();

    $query_params = $request->query->all();
    $rest_view_path = $request->getSchemeAndHttpHost()
      . '/' . ltrim($view->getPath(), '/')
      . '?' . http_build_query($query_params);

    $wrapper_id = 'htmx-view-' . hash('crc32', $view_id . $display_id);
    $build = [
      '#theme' => 'htmx_view_view',
      '#htmx_view' => $this,
      '#view_id' => $view_id,
      '#display_id' => $display_id,
      '#items' => [],
      '#attributes' => [
        'id' => $wrapper_id,
      ],
      '#cache' => [
        'tags' => $view->getCacheTags(),
        'contexts' => Cache::mergeContexts($view->element['#cache']['contexts'] ?? [], ['url', 'languages']),
      ],
    ];

    $response = \Drupal::httpClient()->get($rest_view_path);
    $view_data = Json::decode($response->getBody()->getContents());
    if (!isset($view_data['search_results'])) {
      $view_data = [
        'search_results' => $view_data,
      ];
    }

    $entity_storages = [];
    $view_builders = [];
    $entity_type_manager = $this->entityTypeManager();
    $language_manager = $this->languageManager();
    foreach ($view_data['search_results'] as $delta => $item) {
      [, $entity_type_and_id, $langcode] = explode(':', $item[$this->getMappedKey('id')]);
      [$entity_type, $entity_id] = explode('/', $entity_type_and_id);

      $language = $language_manager->getLanguage($langcode);
      if ($this->getLazyLoadAll()) {
        $lazy_load_all_settings = $this->getLazyLoadAllSettings();
        $trigger = $lazy_load_all_settings['trigger'] ?? 'revealed';
        if ((int) $lazy_load_all_settings['preload'] > 0 && $delta < $lazy_load_all_settings['preload']) {
          $trigger = 'load';
        }

        $build['#items'][] = HtmxEntityPartial::fromParameters($entity_type, $entity_id)
          ->setViewMode($entity_view_mode)
          ->setOption('language', $language)
          ->setHtmxParameter('swap', 'outerHTML')
          ->setHtmxParameter('trigger', $trigger)
          ->render();
      }
      else {
        if (!isset($entity_storages[$entity_type])) {
          $entity_storages[$entity_type] = $entity_type_manager->getStorage($entity_type);
        }
        if (!isset($view_builders[$entity_type])) {
          $view_builders[$entity_type] = $entity_type_manager->getViewBuilder($entity_type);
        }

        $entity = $entity_storages[$entity_type]->load($entity_id);
        $build['#items'][] = $view_builders[$entity_type]->view($entity, $entity_view_mode, $langcode);
      }
    }

    $route_parameters = ['htmx_view' => $this->id()];
    $route_options = ['query' => $query_params];

    $pager_definition = $this->getPagerDefinition();
    if (isset($view_data['pager']) && ($pager = $view_data['pager']) && $pager_definition['type'] !== '_none') {
      $view->initPager();
      $pager_definition['options'] = $view->pager->options ?? [];
      $pager_query_params = $query_params;
      unset($pager_query_params['page']);
      $build['#pager'] = [
        '#theme' => 'htmx_view_pager',
        '#htmx_view' => $this->id(),
        '#definition' => $pager_definition,
        '#extra_htmx_attributes' => [
          'data-hx-target' => "#$wrapper_id",
          'data-hx-swap' => 'outerHTML',
        ],
        '#route' => [
          'name' => 'htmx_extras.lazy_load.htmx_view_plain',
          'parameters' => $route_parameters,
          'options' => ['query' => $pager_query_params],
        ],
        '#current_page' => $pager['current_page'],
        '#total_items' => $pager['total_items'],
        '#total_pages' => $pager['total_pages'],
        '#items_per_page' => $pager['items_per_page'],
        '#htmx_view_wrapper_id' => $wrapper_id,
        '#attributes' => [
          'id' => $wrapper_id . '--pager',
          'class' => array_filter(['htmx-view-pager', $pager_definition['type'] ?? NULL]),
        ],
      ];
    }

    if ($this->showFacets() && ($mapped_facets = $this->getMappedFacets($view_data, $route_parameters, $route_options))) {
      $build['#facets'] = array_map(function ($facet) use ($wrapper_id) {
        $facet_build = [
          '#theme' => 'htmx_view_facet',
          '#facet_config' => $this->getFacetsConfiguration()[$facet['facet_id']] ?? [],
          '#htmx_view' => $this->id(),
        ];

        foreach ($facet as $key => $value) {
          if ($key === 'items') {
            foreach ($value as $delta => $item) {
              $facet_build['#items'][$delta] = $this->renderFacetItem($item, $facet_build['#facet_config'], $wrapper_id);
            }
          }
          else {
            $facet_build["#$key"] = $value;
          }
        }
        return $facet_build;
      }, $mapped_facets);

      if ($this->showFacetSummary()) {
        $facet_summary_items = array_filter(array_map(function ($facet) {
          $facet['items'] = array_filter($facet['items'], static function ($item) {
            return $item['active'] > 0;
          });
          return $facet;
        }, $mapped_facets), static function ($facet) {
          return !empty($facet['items']);
        });
        $build['#facet_summary'] = [
          '#theme' => 'htmx_view_facet_summary',
          '#htmx_view' => $this->id(),
          '#items' => [],
          '#extra_htmx_attributes' => [
            'data-hx-target' => "#$wrapper_id",
            'data-hx-swap' => 'outerHTML',
          ],
        ];

        foreach ($facet_summary_items as $facet_id => $facet) {
          foreach ($facet['items'] as $item) {
            if ($item['active'] === 1) {
              $build['#facet_summary']['#items'][] = [
                'facet_id' => $facet_id,
                'facet_label' => $facet['label'],
                'label' => $item['values']['value'],
                'url' => $item['url'],
                'query_params' => $item['query_params'],
              ];
            }

            foreach ($item['children'][0] ?? [] as $child_item) {
              if (($child_item['values']['active'] ?? NULL) !== 'true') {
                continue;
              }

              $build['#facet_summary']['#items'][] = [
                'facet_id' => $facet_id,
                'facet_label' => $facet['label'],
                'label' => "{$item['values']['value']} - {$child_item['values']['value']}",
                'url' => $child_item['url'],
                'query_params' => $child_item['query_params'],
              ];
            }
          }
        }
      }
    }

    // Add AlpineJS only if needed.
    if (isset($build['#facets']) || isset($build['#pager'])) {
      $build['#attached']['library'][] = 'htmx_extras/alpine';
    }

    return $build;
  }

  private function renderFacetItem(array $item, array $facet_config = [], string $wrapper_id = ''): array {
    $facet_item_build = [
      '#theme' => 'htmx_view_facet_item',
      '#item' => $item,
      '#facet_config' => $facet_config,
      '#extra_htmx_attributes' => [
        'data-hx-target' => "#$wrapper_id",
        'data-hx-swap' => 'outerHTML',
      ],
    ];

    foreach ($item['children'][0] ?? [] as $child_delta => $child_item) {
      $facet_item_build['#children'][$child_delta] = $this->renderFacetItem($child_item, $facet_config, $wrapper_id);
    }

    return $facet_item_build;
  }

  private function getMappedFacets(array $view_data, array $route_parameters, array $route_options): array {
    if (!isset($view_data['facets_metadata'], $view_data['facets'])) {
      return [];
    }

    $mapped_facets = [];
    foreach ($view_data['facets_metadata'] as $key => $facet) {
      $mapped_facets[$key] = $facet + ['facet_id' => $key];

      $results = array_filter($view_data['facets'], function ($items) use ($facet) {
        return isset($items[0][$facet['field_id']]);
      });
      $mapped_facets[$key]['items'] = reset($results)[0][$facet['field_id']] ?? [];

      foreach ($mapped_facets[$key]['items'] as &$item) {
        $this->refactorFacetItem($item, $route_parameters, $route_options);
      }
    }

    // Sort the facets by weight.
    uasort($mapped_facets, function ($a, $b) {
      return $a['weight'] <=> $b['weight'];
    });

    // Remove facets w/o items.
    return array_filter($mapped_facets, function ($facet) {
      return !empty($facet['items']);
    });
  }

  private function refactorFacetItem(array &$item, array $route_parameters, array $route_options): void {
    $route_parameters_copy = $route_parameters;
    $route_options_copy = $route_options;

    // Retrieve the full facet URL.
    $url = $item['url'];
    // Filter everything out aside from the query parameters.
    $query_params_str = parse_url($url, PHP_URL_QUERY) ?: '';
    // Parse the query params string to an array.
    parse_str($query_params_str, $query_params);
    // Remove query params that we don't want.
    unset($query_params['ajax_page_state'], $query_params['page']);
    // Rebuild the query params to a string.
    $item['query_params'] = http_build_query($query_params);
    $route_options['query'] = $query_params;
    $item['url'] = Url::fromRoute('htmx_extras.lazy_load.htmx_view_plain', $route_parameters, $route_options)->toString();
    $item['active'] = ($item['values']['active'] ?? NULL) === 'true' ? 1 : 0;

    if (!empty($item['children'][0])) {
      foreach ($item['children'][0] as &$child_item) {
        $this->refactorFacetItem($child_item, $route_parameters_copy, $route_options_copy);
        if (($child_item['values']['active'] ?? NULL) === 'true' && ($item['active'] ?? 0) < 1) {
          $item['active'] = 2;
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE) {
    parent::postSave($storage, $update);
    \Drupal::service('router.builder')->setRebuildNeeded();
  }

}
