<?php

declare(strict_types=1);

namespace Drupal\htmx_extras\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Ajax\ScrollTopCommand;
use Drupal\Core\Entity\Entity\EntityViewMode;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Theme\ComponentPluginManager;
use Drupal\htmx_extras\Entity\HtmxView;
use Drupal\views\Entity\View;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * HTMX View form.
 */
class HtmxViewForm extends EntityForm {

  private ComponentPluginManager $sdcManager;

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->sdcManager = $container->get('plugin.manager.sdc');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);
    /** @var \Drupal\htmx_extras\HtmxViewInterface $entity */
    $entity = $this->entity;

    $form['#attached']['library'][] = 'core/drupal.ajax';

    $form['_tabs'] = [
      '#type' => 'vertical_tabs',
      '#weight' => -10,
    ];

    // ----- General -----
    $form['_general'] = [
      '#type' => 'details',
      '#title' => $this->t('General'),
      '#open' => TRUE,
      '#group' => '_tabs',
    ];

    $form['_general']['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Admin label'),
      '#maxlength' => 255,
      '#default_value' => $entity->label(),
      '#required' => TRUE,
    ];

    $form['_general']['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->id(),
      '#machine_name' => [
        'exists' => [HtmxView::class, 'load'],
        'source' => ['_general', 'label'],
      ],
      '#disabled' => !$entity->isNew(),
    ];

    $form['_general']['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $entity->status(),
    ];

    $form['_general']['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $entity->get('description'),
      '#rows' => 2,
    ];

    $form['_general']['view_source'] = [
      '#type' => 'select',
      '#title' => $this->t('View source'),
      '#description' => $this->t('Where to get the results from (should be a rest view)'),
      '#empty_option' => $this->t('- Select -'),
      '#options' => [],
      '#default_value' => $entity->get('view_source'),
      '#required' => TRUE,
    ];
    $views = View::loadMultiple();
    foreach ($views as $view) {
      foreach ($view->__get('display') as $display_id => $display_definition) {
        if ($display_definition['display_plugin'] !== 'rest_export') {
          continue;
        }

        $form['_general']['view_source']['#options'][$view->id()]["{$view->id()}:$display_id"]
          = $view->label() . ' | ' . $display_definition['display_title'];
      }
    }

    $form['_general']['mapped_keys'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Mapped field keys'),
      '#description' => $this->t('The rest view requires at least the Search API "Item URL" and "Item ID"-field values.'),
      '#tree' => TRUE,
    ];

    $form['_general']['mapped_keys']['id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Item ID'),
      '#default_value' => $entity->getMappedKey('id'),
      '#placeholder' => 'id',
      '#required' => TRUE,
    ];

    // ----- Display -----
    $form['_display'] = [
      '#type' => 'details',
      '#title' => $this->t('Display'),
      '#open' => FALSE,
      '#group' => '_tabs',
    ];

    $view_modes = EntityViewMode::loadMultiple();
    $view_mode_options = [];
    foreach ($view_modes as $view_mode) {
      $view_mode_id = str_replace($view_mode->getTargetType() . '.', '', $view_mode->id());
      $view_mode_options[$view_mode->getTargetType()][$view_mode_id] = $view_mode->label();
    }
    $form['_display']['entity_view_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity view mode'),
      '#options' => $view_mode_options,
      '#default_value' => $entity->getEntityViewMode(),
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
    ];

    $form['_display']['view_type'] = [
      '#type' => 'select',
      '#title' => $this->t('View type'),
      '#options' => [
        'page' => $this->t('Page'),
        'block' => $this->t('Block'),
      ],
      '#required' => TRUE,
      '#default_value' => $entity->getViewType(),
    ];

    $form['_display']['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title'),
      '#maxlength' => 255,
      '#default_value' => $entity->getTitle(),
      '#required' => TRUE,
    ];

    $form['_display']['path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Path'),
      '#default_value' => $entity->getPath(),
      '#placeholder' => '/path/to/view',
      '#states' => [
        'visible' => [
          ':input[name="view_type"]' => ['value' => 'page'],
        ],
        'required' => [
          ':input[name="view_type"]' => ['value' => 'page'],
        ],
      ],
    ];

    $form['_display']['lazy_load_all'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Lazy load all'),
      '#description' => $this->t('If enabled, next to the view itself also every single item will be lazy loaded. This can cause the page to "jump" around more if the partial skeleton isn\'t properly styled.'),
      '#default_value' => $entity->getLazyLoadAll(),
    ];

    $lazy_load_all_settings = $entity->getLazyLoadAllSettings();
    $form['_display']['lazy_load_all_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Lazy load settings'),
      '#tree' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="lazy_load_all"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['_display']['lazy_load_all_settings']['trigger'] = [
      '#type' => 'radios',
      '#title' => $this->t('HTMX Trigger'),
      '#description' => $this->t('Use "intersect once" instead of "revealed" when working with overflowed divs, for example horizontal scrolling.'),
      '#options' => [
        'revealed' => 'revealed',
        'intersect once' => 'intersect once',
        'load' => 'load',
      ],
      '#default_value' => $lazy_load_all_settings['trigger'] ?? 'revealed',
    ];

    $form['_display']['lazy_load_all_settings']['preload'] = [
      '#type' => 'number',
      '#title' => $this->t('Preload items'),
      '#description' => $this->t('Preload a number of items on load, to reduce waiting times / the page jumping around. Use "0" to lazy load all with the above trigger.'),
      '#default_value' => $lazy_load_all_settings['preload'] ?? 0,
      '#states' => [
        'visible' => [
          ':input[name="lazy_load_all_settings[trigger]"]' => ['!value' => 'load'],
        ],
      ],
    ];

    // ----- Pager -----
    $form['_pager'] = [
      '#type' => 'details',
      '#title' => $this->t('Pager'),
      '#open' => FALSE,
      '#group' => '_tabs',
    ];

    $form['_pager']['pager'] = [
      '#type' => 'fieldset',
      '#tree' => TRUE,
    ];

    $form['_pager']['pager']['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Type'),
      '#options' => [
        '_none' => $this->t(' - None - '),
        'full' => $this->t('Full'),
        'mini' => $this->t('Mini'),
        'infinite_scroll' => $this->t('Infinite scroll'),
        'load_more' => $this->t('Load more'),
      ],
      '#required' => TRUE,
      '#default_value' => $entity->getPagerDefinition()['type'] ?? NULL,
    ];

    $form['_pager']['pager']['scroll_to_top'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Scroll to top when changing page'),
      '#default_value' => $entity->getPagerDefinition()['scroll_to_top'] ?? TRUE,
      '#states' => [
        'visible' => [
          ':input[name="pager[type]"]' => [
            ['value' => 'full'],
            'or',
            ['value' => 'mini'],
          ],
        ],
      ],
    ];

    $form['_pager']['pager']['hide_pager'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide pager if only one page is available'),
      '#default_value' => $entity->getPagerDefinition()['hide_pager'] ?? FALSE,
      '#states' => [
        'visible' => [
          ':input[name="pager[type]"]' => ['!value' => '_none'],
        ],
      ],
    ];

    $form['_pager']['pager']['_markup'] = [
      '#markup' => '<em>' . $this->t('Other pager settings should be configured in the source view.') . '</em>',
    ];

    // ----- Facets -----
    $form['_facets'] = [
      '#type' => 'details',
      '#title' => $this->t('Facets'),
      '#open' => FALSE,
      '#group' => '_tabs',
    ];

    $module_handler = \Drupal::moduleHandler();
    if (!$module_handler->moduleExists('facets') || $entity->isNew()) {
      $form['_facets']['info'] = [
        '#markup' => $this->t("Please, come back after you've saved the HTMX view."),
      ];
    }
    else {
      $form['_facets']['show_facets'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Show facets'),
        '#default_value' => $entity->showFacets(),
      ];

      $form['_facets']['show_facet_summary'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Show facet summary'),
        '#default_value' => $entity->showFacetSummary(),
        '#states' => [
          'visible' => [
            ':input[name="show_facets"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['_facets']['facets'] = [
        '#type' => 'fieldset',
        '#tree' => TRUE,
        '#states' => [
          'visible' => [
            ':input[name="show_facets"]' => ['checked' => TRUE],
          ],
        ],
      ];
      try {
        /** @var \Drupal\facets\FacetManager\DefaultFacetManager $facet_manager */
        $facet_manager = \Drupal::service('facets.manager');
        $view = $entity->getViewSource();
        $facets = $facet_manager->getFacetsByFacetSourceId("search_api:views_rest__{$view->id()}__$view->current_display");
        foreach ($facets as $facet_id => $facet) {
          $form['_facets']['facets'][$facet_id] = [
            '#type' => 'details',
            '#title' => $facet->label(),
          ];

          $form['_facets']['facets'][$facet_id]['show_title'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Show facet title'),
            '#default_value' => $entity->getFacetsConfiguration()[$facet_id]['show_title'] ?? TRUE,
          ];

          $form['_facets']['facets'][$facet_id]['show_count'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Show item count'),
            '#default_value' => $entity->getFacetsConfiguration()[$facet_id]['show_count'] ?? TRUE,
          ];
        }
      }
      catch (\Exception $e) {}
    }

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function actions(array $form, FormStateInterface $form_state): array {
    $actions = parent::actions($form, $form_state);
    foreach ($actions as &$action) {
      if (!in_array($action['#type'] ?? NULL, ['button', 'submit'])) {
        continue;
      }

      $action['#ajax'] = [
        'callback' => '::generalAjaxCallback',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Saving') . '...',
        ],
      ];
    }

    return $actions;
  }

  public function generalAjaxCallback(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    foreach ($triggering_element['#submit'] as $handler) {
      if (is_string($handler) && str_starts_with($handler, '::')) {
        $handler = [$this, substr($handler, 2)];
      }
      call_user_func_array($handler, [&$form, $form_state]);
    }

    $response = new AjaxResponse();
    $messages = $this->messenger()->all();
    $this->messenger()->deleteAll();
    foreach ($messages as $type => $type_messages) {
      foreach ($type_messages as $message) {
        $response->addCommand(new MessageCommand($message, NULL, ['type' => $type]));
      }
    }
    $response->addCommand(new ScrollTopCommand('.messages-list'));
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $this->entity->label()];
    $this->messenger()->addStatus(
      match($result) {
        \SAVED_NEW => $this->t('Created new HTMX View %label.', $message_args),
        \SAVED_UPDATED => $this->t('Updated HTMX View %label.', $message_args),
      }
    );
    $form_state->setRedirectUrl($this->entity->toUrl('edit-form'));
    return $result;
  }

}
